/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * 知启蒙WEB容器（zhiqim_httpd）在LGPL3.0协议下开源：https://www.zhiqim.com/gitcan/zhiqim/zhiqim_httpd.htm
 *
 * This file is part of [zhiqim_httpd].
 * 
 * [zhiqim_httpd] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_httpd] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_httpd].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.httpd;

import java.io.IOException;
import java.io.InputStream;

import org.zhiqim.kernel.constants.CodeConstants;
import org.zhiqim.kernel.util.Asserts;
import org.zhiqim.kernel.util.Bytes;
import org.zhiqim.kernel.util.Streams;
import org.zhiqim.kernel.util.Strings;
import org.zhiqim.kernel.util.mutables.MInt;

/**
 * HttpWebsocket数据格式：
 * 第一字节，第1位：FIN，标识是否为此消息的最后一个数据包
 * 第一字节，第5,6,7,8位，opcode，操作码（0－15）：
 * 0x00：标识一个中间数据包
 * 0x01：标识一个text类型数据包
 * 0x02：标识一个binary类型数据包
 * 0x08：标识一个断开连接类型数据包
 * 0x09：标识一个ping类型数据包
 * 0x0A：表示一个pong类型数据包
 * 0x03-07：保留
 * 0x0B-0F：保留
 * 
 * 0–999        保留段, 未使用.
 * 1000         CLOSE_NORMAL            正常关闭; 无论为何目的而创建, 该链接都已成功完成任务.
 * 1001         CLOSE_GOING_AWAY        终端离开, 可能因为服务端错误, 也可能因为浏览器正从打开连接的页面跳转离开.
 * 1002         CLOSE_PROTOCOL_ERROR    由于协议错误而中断连接.
 * 1003         CLOSE_UNSUPPORTED       由于接收到不允许的数据类型而断开连接 (如仅接收文本数据的终端接收到了二进制数据).
 * 1004                                 保留. 其意义可能会在未来定义.
 * 1005         CLOSE_NO_STATUS         保留.  表示没有收到预期的状态码.
 * 1006         CLOSE_ABNORMAL          保留. 用于期望收到状态码时连接非正常关闭 (也就是说, 没有发送关闭帧).
 * 1007         Unsupported Data        由于收到了格式不符的数据而断开连接 (如文本消息中包含了非 UTF-8 数据).
 * 1008         Policy Violation        由于收到不符合约定的数据而断开连接. 这是一个通用状态码, 用于不适合使用 1003 和 1009 状态码的场景.
 * 1009         CLOSE_TOO_LARGE         由于收到过大的数据帧而断开连接.
 * 1010         Missing Extension       客户端期望服务器商定一个或多个拓展, 但服务器没有处理, 因此客户端断开连接.
 * 1011         Internal Error          客户端由于遇到没有预料的情况阻止其完成请求, 因此服务端断开连接.
 * 1012         Service Restart         服务器由于重启而断开连接. [Ref]
 * 1013         Try Again Later         服务器由于临时原因断开连接, 如服务器过载因此断开一部分客户端连接. [Ref]
 * 1014                                 由 WebSocket 标准保留以便未来使用.
 * 1015         TLS Handshake           保留. 表示连接由于无法完成 TLS 握手而关闭 (例如无法验证服务器证书).
 * 1016–1999                            由 WebSocket 标准保留以便未来使用.
 * 2000–2999                            由 WebSocket 拓展保留使用.
 * 3000–3999                            可以由库或框架使用. 不应由应用使用. 可以在 IANA 注册, 先到先得.
 * 4000–4999                            可以由应用使用.
 *
 * @version v1.0.0 @author zouzhigang 2017-6-1 新建与整理
 */
public class HttpWebsocketMessage implements CodeConstants
{
    protected boolean fin;
    protected int opcode;
    
    protected boolean mask;
    protected byte[] masks;
    protected int length;
    protected byte[] content;
    
    //内容解析属性：真实内容长度&内容文本
    protected int contentLen;
    protected String text;
    
    public HttpWebsocketMessage parse(InputStream in) throws IOException
    {
        int first = in.read() & 0xFF;
        int second = in.read() & 0xFF;
        
        //1.FIN
        this.fin = (first & 0x80) != 0;
        
        //2.opcode
        this.opcode = first & 0x0F;
        
        //3.MASK
        this.mask = (second & 0x80) != 0;
        
        //4.length
        this.contentLen = this.length = second & 0x7F;
        if (this.length == 126)
        {//2字节表示长度
            byte[] buf = Streams.getBytes(in, 2);
            this.contentLen = Bytes.BU.getShortUnsigned(buf, 0);
        }
        else if (this.length == 127)
        {//8字节表示长度
            byte[] buf = Streams.getBytes(in, 8);
            long len = Bytes.BU.getLong(buf, 0);
            if (len > Integer.MAX_VALUE)
                throw Asserts.exception("超出支持的能力");
            
            this.contentLen = (int)len;
        }
        
        //5.masks
        if (mask)
        {//存在掩码的情况，取4字节
            this.masks = Streams.getBytes(in, 4);
        }
        
        //6.内容
        if (contentLen > 0)
        {//有内容
            this.content = Streams.getBytes(in, contentLen);
            if (mask)
            {//存在掩码的情况，作处理
                for (int i=0;i<content.length;i++)
                {
                    content[i] = (byte)(content[i] ^ masks[i % 4]);
                }
            }
            
            if (isText())
            {//文本的解析成字符串
                text = Strings.newStringUTF8(content);
            }
            else if (isClose())
            {//结束符的有状态码，CloseEvent.code，如1001
                text = String.valueOf(Bytes.BU.getShortUnsigned(content, 0));
            }
        }
        
        return this;
    }
    
    public HttpWebsocketMessage build(String text)
    {
        this.fin = true;
        this.opcode = 0x01;
        this.mask = false;
        
        this.text = text;
        this.content = text.getBytes(_UTF_8_C_);
        this.contentLen = this.content.length;
        
        //计算长度
        if (this.contentLen < 126)
            this.length = contentLen;
        else if (this.contentLen <= 65535)
            this.length = 126;
        else
            this.length = 127;

        return this;
    }
    
    public HttpWebsocketMessage buildPing()
    {
        this.fin = true;
        this.opcode = 0x09;
        this.mask = false;
        
        this.text = null;
        this.content = null;
        this.contentLen = 0;
        
        this.length = 0;
        
        return this;
    }
    
    public HttpWebsocketMessage buildPong()
    {
        this.fin = true;
        this.opcode = 0x0A;
        this.mask = false;
        
        this.text = null;
        this.content = null;
        this.contentLen = 0;
        
        this.length = 0;
        
        return this;
    }
    
    public byte[] toBytes()
    {//不加masks，长度＝前两字节+可能的长度字节+内容
        int len = 2 + ((this.length == 126)?2:(this.length == 127)?8:0) + contentLen;
        byte[] buf = new byte[len];
        
        buf[0] = (byte)(fin?(0x80 | opcode):opcode);
        buf[1] = (byte)this.length;
        
        MInt off = new MInt(2);
        if (length == 126)
            Bytes.BU.putShort(buf, off, contentLen);
        else if (length == 127)
            Bytes.BU.putLong(buf, off, contentLen);
        
        if (contentLen > 0)
        {//有内容
            Bytes.putBytes(buf, off, content);
        }
        
        return buf;
    }
    
    /***********************************************************************/
    // 业务方法
    /***********************************************************************/
    
    public boolean isText()
    {
        return opcode == 0x01;
    }
    
    public boolean isBinary()
    {
        return opcode == 0x02;
    }
    
    public boolean isClose()
    {
        return opcode == 0x08;
    }
    
    public boolean isPing()
    {
        return opcode == 0x09;
    }
    
    public boolean isPong()
    {
        return opcode == 0x0A;
    }
    
    /***********************************************************************/
    // 获取&设置
    /***********************************************************************/
    
    public boolean isFin()
    {
        return fin;
    }

    public int getOpcode()
    {
        return opcode;
    }

    public boolean isMask()
    {
        return mask;
    }

    public byte[] getMasks()
    {
        return masks;
    }

    public int getLength()
    {
        return length;
    }

    public String getText()
    {
        return text;
    }
    
    public int getContentLen()
    {
        return contentLen;
    }
    
    public byte[] getContent()
    {
        return (contentLen == 0)?new byte[0]:content;
    }
}
