/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * 知启蒙WEB容器（zhiqim_httpd）在LGPL3.0协议下开源：https://www.zhiqim.com/gitcan/zhiqim/zhiqim_httpd.htm
 *
 * This file is part of [zhiqim_httpd].
 * 
 * [zhiqim_httpd] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_httpd] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_httpd].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.httpd;

import java.lang.reflect.Method;

import org.zhiqim.kernel.Global;
import org.zhiqim.kernel.extend.HashMapSO;
import org.zhiqim.kernel.util.Asserts;
import org.zhiqim.kernel.util.Classes;
import org.zhiqim.kernel.util.Ints;
import org.zhiqim.kernel.util.Longs;
import org.zhiqim.kernel.util.Strings;
import org.zhiqim.kernel.util.Types;

/**
 * HTTP在线用户定义，子类继承时，需设置到HttpSessionManager中
 *
 * @version v1.0.0 @author zouzhigang 2014-3-21 新建与整理
 */
public abstract class HttpSessionUser
{
    protected transient HttpSession session;
    
    //两个冗余的字段，用于生成JSON时可以通过字符串找到对应的类名和字段
    protected String className;
    protected String sessionName;
    
    //通用的值表，和session中的attributeMap不同的是：
    //1.valueMap绑定sessionUser，解绑即失效
    //2.attributeMap可用在和sessionUser无关的地方
    protected HashMapSO valueMap = new HashMapSO();
    
    public HttpSessionUser()
    {
        className = this.getClass().getName();
    }
    
    /**************************************************/
    //要求子类强制实现的方法
    /**************************************************/
    
    public abstract String getSessionName();
    
    /**************************************************/
    //由HttpSession回调
    /**************************************************/
    
    public void setSession(HttpSession session)
    {
        this.session = session;
        this.sessionName = getSessionName();
    }
    
    /**************************************************/
    //获取HttpSession对象和相关属性
    /**************************************************/
    
    public HttpSession getSession()
    {
        return session;
    }
    
    public HttpContext getContext()
    {
        return session.getContext();
    }
    
    public String getSessionId()
    {
        return session.getId();
    }
    
    public boolean isNew()
    {
        return session.isNew();
    }
    
    public String getSessionIp()
    {
        return session.getSessionIp();
    }
    
    public String getUserAgent()
    {
        return session.getUserAgent();
    }
    
    public long getCreated()
    {
        return session.getCreated();
    }
    
    public String getCreateTime()
    {
        return session.getCreateTime();
    }
    
    public long getLastAccessed()
    {
        return session.getLastAccessed();
    }
    
    public String getLastAccessTime()
    {
        return session.getLastAccessTime();
    }
    
    public long getRemainSecond()
    {
        return session.getRemainSecond();
    }
    
    public long getRemainMinute()
    {
        return session.getRemainMinute();
    }
    
    public long[] getRemainTime()
    {
        return session.getRemainTime();
    }
    
    public String getRemainTimeDesc()
    {
        return session.getRemainTimeDesc();
    }
    
    /**********************************************/
    //读写关联到HttpSession中attribute列表
    /**********************************************/
    
    public HashMapSO getAttributes()
    {
        return session.getAttributes();
    }
    
    public Object getAttribute(String name)
    {
        return session.getAttribute(name);
    }
    
    public Object removeAttribute(String name)
    {
        return session.removeAttribute(name);
    }
    
    public void setAttribute(String name, Object value)
    {
        session.setAttribute(name, value);
    }
    
    public boolean hasAttribute(String name)
    {
        return session.hasAttribute(name);
    }

    /**********************************************/
    //读写关联到HttpSessionUser自己的参数列表
    /**********************************************/
    
    public HashMapSO getValues()
    {
        return valueMap;
    }
    
    public Object getValue(String key)
    {
        return valueMap.get(key);
    }
    
    public boolean hasValue(String key)
    {
        return valueMap.containsKey(key);
    }
    
    public Object removeValue(String key)
    {
        return valueMap.remove(key);
    }
    
    public void setValue(String key, Object value)
    {
        valueMap.put(key, value);
    }

    public void setValueTrue(String key)
    {
        valueMap.put(key, true);
    }
    
    public void setValueFalse(String key)
    {
        valueMap.put(key, false);
    }
    
    public String getValueString(String key)
    {
        return getValueString(key, null);
    }
    
    public String getValueString(String key, String defaultValue)
    {
        Object value = getValue(key);
        return (value == null)?defaultValue:Strings.trim(String.valueOf(value));
    }
    
    public int getValueInt(String key)
    {
        return getValueInt(key, -1);
    }
    
    public int getValueInt(String key, int defaultValue)
    {
        return Ints.toInt(valueMap.get(key), defaultValue);
    }
    
    public long getValueLong(String key)
    {
        return getValueLong(key, -1);
    }
    
    public long getValueLong(String key, long defaultValue)
    {
        return Longs.toLong(valueMap.get(key), defaultValue);
    }
    
    public boolean getValueBoolean(String key, boolean defaultValue)
    {
        Object value = valueMap.get(key);
        if (value == null)
            return defaultValue;
        else if (value instanceof Boolean)
            return (Boolean)value;
        else if (value instanceof String)
            return Boolean.parseBoolean((String)value);
        else
            return defaultValue;
    }
    
    /**
     * 有条件判断是否有值，如果valueMap有值则返回valueMap中的true/false，如果没有，则通过类和静态方法名，返回true/false
     * 注意：1.如果valueMap中有值，值不是boolean类型，返回false
     *      2.如果类和静态方法返回值不是boolean类型，返回false，如果不是静态方法或静态方法参数不是HttpSessionUser返回false
     * 
     * @param key           键值
     * @param className     类名
     * @param methodName    方法名，方法必须是静态方法，且参数为HttpSessionUser实例，如public static boolean isOpen(HttpSessionUser sessionUser){};
     * @return              =true/false
     */
    public boolean isValue(String key, String className, String staticMethodName)
    {
        if (hasValue(key))
        {//有值则取值
            Object value = getValue(key);
            return Types.isBoolean(value)?(Boolean)value:false;
        }
        
        Class<?> clazz = Global.forName(className);
        Asserts.as(clazz != null?null:"判断会话属性值时，传入的类名或类别名不正确");
        
        Method method = Classes.getMethodDeep(clazz, staticMethodName, this.getClass());
        Asserts.as(method != null?null:"判断会话属性值时，传入的静态方法名未找到");
        Asserts.as(Classes.isStaticMethod(method)?null:"判断会话属性值时，传入的方法名不是静态方法");
        
        try
        {
            Object value = method.invoke(null, this);
            Asserts.as(Types.isBoolean(value)?null:"判断会话属性值时，方法返回的值不是boolean型");
            
            return (Boolean)value;
        }
        catch (Exception e)
        {//异常返回false，表示业务处理失败
            return false;
        }
    }
}
