/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[欢迎加盟知启蒙，一起邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_manager.htm
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.zhiqim.manager.dao;

import java.sql.SQLException;
import java.util.LinkedHashMap;
import java.util.List;

import org.zhiqim.httpd.HttpContext;
import org.zhiqim.kernel.annotation.AnAlias;
import org.zhiqim.kernel.config.ItemType;
import org.zhiqim.kernel.constants.SignConstants;
import org.zhiqim.kernel.util.Ints;
import org.zhiqim.kernel.util.Validates;
import org.zhiqim.manager.ZmrBootstrap;
import org.zhiqim.manager.ZmrConstants;
import org.zhiqim.manager.dbo.ZmrParam;
import org.zhiqim.orm.ORMException;
import org.zhiqim.orm.ZTable;
import org.zhiqim.orm.dbo.Selector;
import org.zhiqim.orm.dbo.Updater;

/**
 * 参数访问对象
 *
 * @version v1.0.0 @author zouzhigang 2015-5-28 新建与整理
 * @version v1.4.1 @author zouzhigang 2018-5-3 由ZmrParamCache改为ZmrParamDao
 */
@AnAlias("ZmrParamDao")
public class ZmrParamDao implements ZmrConstants, SignConstants
{
    /**************************************************/
    //新建参数对象
    /**************************************************/
    
    /** 新建参数对象，默认值和验证为空，允许修改 */
    public static void addParam(HttpContext context, String paramGroup, String paramKey, int paramSeq, String paramDesc) throws Exception
    {
        addParam(context, paramGroup, paramKey, null, paramSeq, ItemType.PUBLIC, null, paramDesc);
    }
    
    /** 新建参数对象，默认验证为空 */
    public static void addParam(HttpContext context, String paramGroup, String paramKey, String paramValue, int paramSeq, ItemType paramType, String paramDesc) throws Exception
    {
        addParam(context, paramGroup, paramKey, paramValue, paramSeq, paramType, null, paramDesc);
    }
    
    /** 新建参数对象，全7个字段 */
    public static void addParam(HttpContext context, String paramGroup, String paramKey, String paramValue, int paramSeq, ItemType paramType, String paramValidate, String paramDesc) throws Exception
    {
        if (ZmrBootstrap.table(context).count(ZmrParam.class, paramGroup, paramKey) == 0)
        {
            ZmrParam param = new ZmrParam();
            param.setParamGroup(paramGroup);
            param.setParamKey(paramKey);
            param.setParamValue(paramValue);
            param.setParamSeq(paramSeq);
            param.setParamType(ItemType.toString(paramType));
            param.setParamValidate(paramValidate);
            param.setParamDesc(paramDesc);

            ZmrBootstrap.table(context).insert(param);
        }
    }
    
    /**************************************************/
    //判断和获取参数
    /**************************************************/
    
    /** 判断是否存在参数组 */
    public static boolean hasGroup(HttpContext context, String paramGroup) throws ORMException, SQLException
    {
        return ZmrBootstrap.table(context).count(ZmrParam.class, new Selector("paramGroup", paramGroup)) > 0;
    }
    
    
    /** 判断是否存在参数 */
    public static boolean hasParam(HttpContext context, String paramGroup, String paramKey) throws ORMException, SQLException
    {
        return ZmrBootstrap.table(context).count(ZmrParam.class, paramGroup, paramKey) > 0;
    }
    
    /** 获取参数组表 */
    public static LinkedHashMap<String, LinkedHashMap<String, ZmrParam>> getGroupMap(HttpContext context) throws ORMException, SQLException
    {
        LinkedHashMap<String, LinkedHashMap<String, ZmrParam>> groupMap = new LinkedHashMap<>();
        
        List<ZmrParam> paramList = ZmrBootstrap.table(context).list(ZmrParam.class, new Selector().addMustNotEqual("paramType", "private").addOrderbyAsc("paramGroup,paramSeq"));
        for (ZmrParam param : paramList)
        {
            LinkedHashMap<String, ZmrParam> paramMap = groupMap.get(param.getParamGroup());
            if (paramMap == null)
            {
                paramMap = new LinkedHashMap<String, ZmrParam>();
                groupMap.put(param.getParamGroup(), paramMap);
            }
            
            paramMap.put(param.getParamKey(), param);
        }
        return groupMap;
    }
    
    /** 获取参数 */
    public static ZmrParam getParam(HttpContext context, String paramGroup, String paramKey) throws ORMException, SQLException
    {
        ZTable table = ZmrBootstrap.table(context);
        return table.item(ZmrParam.class, paramGroup, paramKey);
    }
    
    /** 获取字符串数据 */
    public static String getString(HttpContext context, String paramGroup, String paramKey) throws ORMException, SQLException
    {
        ZmrParam param = getParam(context, paramGroup, paramKey);
        return (param == null)?null:param.getParamValue();
    }
    
    /** 获取字符串数据，为null取缺省值 */
    public static String getString(HttpContext context, String paramGroup, String paramKey, String defaultValue) throws ORMException, SQLException
    {
        String value = getString(context, paramGroup, paramKey);
        return (Validates.isEmptyBlank(value))?defaultValue:value;
    }
    
    /** 获取整型数据 */
    public static int getInt(HttpContext context, String paramGroup, String paramKey) throws ORMException, SQLException
    {
        return getInt(context, paramGroup, paramKey, -1);
    }
    
    /** 获取整型数据，为null取缺省值 */
    public static int getInt(HttpContext context, String paramGroup, String paramKey, int defaultValue) throws ORMException, SQLException
    {
        return Ints.toInt(getString(context, paramGroup, paramKey), defaultValue);
    }
    
    /** 判断是否为真 */
    public static boolean isTrue(HttpContext context, String paramGroup, String paramKey) throws ORMException, SQLException
    {
        return _TRUE_.equals(getString(context, paramGroup, paramKey));
    }
    
    /** 判断是否为真，为null取缺省值 */
    public static boolean isTrue(HttpContext context, String paramGroup, String paramKey, String defaultValue) throws ORMException, SQLException
    {
        return _TRUE_.equals(getString(context, paramGroup, paramKey, defaultValue));
    }

    /**************************************************/
    //更新和刷新参数
    /**************************************************/
    
    /** 更新缓存值 */
    public static boolean doUpdate(HttpContext context, String paramGroup, String paramKey, int paramValue) throws ORMException, SQLException
    {
        return doUpdate(context, paramGroup, paramKey, String.valueOf(paramValue));
    }
    
    /** 更新缓存值 */
    public static boolean doUpdate(HttpContext context, String paramGroup, String paramKey, long paramValue) throws ORMException, SQLException
    {
        return doUpdate(context, paramGroup, paramKey, String.valueOf(paramValue));
    }
    
    /** 更新缓存值 */
    public static boolean doUpdate(HttpContext context, String paramGroup, String paramKey, boolean paramValue) throws ORMException, SQLException
    {
        return doUpdate(context, paramGroup, paramKey, String.valueOf(paramValue));
    }
    
    /** 更新缓存值 */
    public static boolean doUpdate(HttpContext context, String paramGroup, String paramKey, String paramValue) throws ORMException, SQLException
    {
        //更新到数据库
        Updater updater = new Updater();
        updater.addMust("paramGroup", paramGroup);
        updater.addMust("paramKey", paramKey);
        updater.addField("paramValue", paramValue);
        return ZmrBootstrap.table(context).update(ZmrParam.class, updater) > 0;
    }
    
    /**************************************************/
    //以下为ZMR本身自带的六项参数
    /**************************************************/
    
    /** 获取首页主题风格 */
    public static String getThemeIndex(HttpContext context) throws ORMException, SQLException
    {
        return getString(context, ZMR_NAME, ZMR_THEME_INDEX, ZMR_THEME_DEFAULT);
    }
    
    /** 更新首页主题风格 */
    public static void doUpdateThemeIndex(HttpContext context, String theme) throws ORMException, SQLException
    {
        doUpdate(context, ZMR_NAME, ZMR_THEME_INDEX, theme);
    }
    
    /** 获取主页主题风格 */
    public static String getThemeMain(HttpContext context) throws ORMException, SQLException
    {
        return getString(context, ZMR_NAME, ZMR_THEME_MAIN, ZMR_THEME_DEFAULT);
    }
    
    /** 更新主页主题风格 */
    public static void doUpdateThemeMain(HttpContext context, String theme) throws ORMException, SQLException
    {
        doUpdate(context, ZMR_NAME, ZMR_THEME_MAIN, theme);
    }
    
    public static boolean isThemeFrame(HttpContext context) throws ORMException, SQLException
    {
        return isTrue(context, ZMR_NAME, ZMR_THEME_FRAME, _FALSE_);
    }
    
    /** 是否有管理登录页 */
    public static boolean hasManageLogin(HttpContext context) throws ORMException, SQLException
    {
        return isTrue(context, ZMR_NAME, ZMR_HAS_MANAGE_LOGIN);
    }
    
    /** 是否有验证码 */
    public static boolean hasVerificationCode(HttpContext context) throws ORMException, SQLException
    {
        return isTrue(context, ZMR_NAME, ZMR_HAS_VERIFICATION_CODE);
    }
    
    /** 是否记住用户名 */
    public static boolean hasRememberCode(HttpContext context) throws ORMException, SQLException
    {
        return isTrue(context, ZMR_NAME, ZMR_HAS_REMEMBER_CODE);
    }
    
    /** 是否记住密码 */
    public static boolean hasRememberPass(HttpContext context) throws ORMException, SQLException
    {
        return isTrue(context, ZMR_NAME, ZMR_HAS_REMEMBER_PASS);
    }
    
    /** 获取主页地址 */
    public static String getMainUrl(HttpContext context) throws ORMException, SQLException
    {
        return getString(context, ZMR_NAME, ZMR_MAIN_URL, ZMR_MAIN_URL_DEFAULT);
    }
    
    /** 获取主页首页地址 */
    public static String getMainIndexUrl(HttpContext context) throws ORMException, SQLException
    {
        return isThemeFrame(context)?ZMR_MAIN_URL_WELCOME:ZMR_MAIN_URL_DEFAULT;
    }

    /** 获取保留操作日志天数 */
    public static int getRemainLogDay(HttpContext context) throws ORMException, SQLException
    {
        return getInt(context, ZMR_NAME, ZMR_REMAIN_LOG_DAY, 7);
    }
    
    /** 是否取操作员继承的所有部门权限 */
    public static boolean isOperatorDeptAllRule(HttpContext context) throws ORMException, SQLException
    {
        return isTrue(context, ZMR_NAME, ZMR_OPERATOR_DEPT_ALL_RULE);
    }
    
    /** 是否使用CDN路径 */
    public static boolean hasCdnPath(HttpContext context) throws ORMException, SQLException
    {
        return !Validates.isEmptyBlank(getString(context, ZMR_NAME, ZMR_CDN_PATH));
    }
    
    /** 获取CDN路径 */
    public static String getCdnPath(HttpContext context) throws ORMException, SQLException
    {
        return getString(context, ZMR_NAME, ZMR_CDN_PATH);
    }
    
    /** 获取公钥 */
    public static String getPublicKey(HttpContext context) throws ORMException, SQLException
    {
        return getString(context, ZMR_NAME, ZMR_PUBLIC_KEY);
    }
    
    /** 获取私钥 */
    public static String getPrivateKey(HttpContext context) throws ORMException, SQLException
    {
        return getString(context, ZMR_NAME, ZMR_PRIVATE_KEY);
    }
}
