/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_kernel.htm
 *
 * This file is part of [zhiqim_kernel].
 * 
 * [zhiqim_kernel] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_kernel] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_kernel].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.kernel.util;

import java.util.regex.Pattern;

import org.zhiqim.kernel.annotation.AnAlias;
import org.zhiqim.kernel.constants.RegexpConstants;

/**
 * 正则表达式验证类
 *
 * @version v1.0.0 @author zouzhigang 2014-3-21 新建与整理
 */
@AnAlias("Validatex")
public class Validatex implements RegexpConstants
{
    /**
     * 正则表达式验证
     * 
     * @param regexp    正则表达式
     * @param value     需要验证的值
     * @return =true表示成功,=false表示失败
     */
    public static boolean isRegexp(String regexp, String value, boolean allowEmpty)
    {
        if (Validates.isEmpty(value))
            return allowEmpty;
        
        regexp = Strings.trim(regexp, "/", "/");
        return Pattern.matches(regexp, value);
    }
    
    /**
     * 是否是布尔型 true|false
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isBoolean(String value, boolean allowEmpty)
    {
        if (Validates.isEmpty(value))
            return allowEmpty;
        
        return "true".equals(value) || "false".equals(value);
    }
    
    /**
     * 是否是数字
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isNumeric(String value, boolean allowEmpty)
    {
        return isRegexp(NUMERIC, value, allowEmpty);
    }
    
    /**
     * 是否整数，包括(正整数、负整数、0)，支持+-号
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isInteger(String value, boolean allowEmpty)
    {
        return isRegexp(INTEGER, value, allowEmpty);
    }
    
    /**
     * 是否正整数，包括(>0的整数)，不支持+号，全数字
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isIntegerPositive(String value, boolean allowEmpty)
    {
        return isRegexp(INTEGER_P, value, allowEmpty);
    }
    
    /**
     * 是否负整数，包括(<0的整数)，-号开头，再[1-9]，再[0-9]*
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isIntegerNegative(String value, boolean allowEmpty)
    {
        return isRegexp(INTEGER_N, value, allowEmpty);
    }
    
    /**
     * 是否非负整数，包括(>=0的整数)，不支持+号，全数字
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isIntegerNonnegative(String value, boolean allowEmpty)
    {
        return isRegexp(INTEGER_N_N, value, allowEmpty);
    }
    
    /**
     * 是否非正整数，包括(<=0的整数)，负数必须-号开头
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isIntegerNonpositive(String value, boolean allowEmpty)
    {
        return isRegexp(INTEGER_N_P, value, allowEmpty);
    }
    
    /**
     * 是否是字母组合
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isAlphabetic(String value, boolean allowEmpty)
    {
        return isRegexp(ALPHABAT, value, allowEmpty);
    }
    
    /**
     * 是否是大写字母组合
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isAlphabeticUppercase(String value, boolean allowEmpty)
    {
        return isRegexp(ALPHABAT_UPPER, value, allowEmpty);
    }
    
    /**
     * 是否是小写字母组合
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isAlphabeticLowercase(String value, boolean allowEmpty)
    {
        return isRegexp(ALPHABAT_LOWER, value, allowEmpty);
    }
    
    /**
     * 是否是字母和数字组合
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isAlphaNumeric(String value, boolean allowEmpty)
    {
        return isRegexp(ALPHA_NUMERIC, value, allowEmpty);
    }
    
    /**
     * 是否是字母和数字组合，字母开头
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isAlphaNumericPA(String value, boolean allowEmpty)
    {
        return isRegexp(ALPHA_NUMERIC_PA, value, allowEmpty);
    }
    
    /**
     * 是否是字母和数字或下划线组合
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isAlphaNumericUl(String value, boolean allowEmpty)
    {
        return isRegexp(ALPHA_NUMERIC_UL, value, allowEmpty);
    }
    
    /**
     * 是否是字母和数字或点号组合
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isAlphaNumericDot(String value, boolean allowEmpty)
    {
        return isRegexp(ALPHA_NUMERIC_DOT, value, allowEmpty);
    }
    
    /**
     * 是否是字母和数字或(.-_~四个非特殊URL字符)组合
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isAlphaNumericUrl(String value, boolean allowEmpty)
    {
        return isRegexp(ALPHA_NUMERIC_URL, value, allowEmpty);
    }
    
    /**
     * 是否是字母和数字组合，字母或下划线开头，java命名规则
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isAlphaNumericUlPA(String value, boolean allowEmpty)
    {
        return isRegexp(ALPHA_NUMERIC_UL_PA, value, allowEmpty);
    }
    
    /**
     * 是否是字母和数字组合，字母或下划线或$开头，js命名规则
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isAlphaNumericUlDlPA(String value, boolean allowEmpty)
    {
        return isRegexp(ALPHA_NUMERIC_UL_DL_PA, value, allowEmpty);
    }
    
    /**
     * 是否是大写字母和数字组合
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isAlphaUpperNumeric(String value, boolean allowEmpty)
    {
        return isRegexp(ALPHA_UPPER_NUMERIC, value, allowEmpty);
    }
    
    /**
     * 是否是小写字母和数字组合
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isAlphaLowerNumeric(String value, boolean allowEmpty)
    {
        return isRegexp(ALPHA_LOWER_NUMERIC, value, allowEmpty);
    }
    
    /**
     * 是否浮点型，支持多位小数，支持正负号前缀，如-12,-12.3,+12.35,-12.356,12,12.3,12.35,0,0.3,0.35,-0.3,-0.35
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isFloat(String value, boolean allowEmpty)
    {
        return isRegexp(FLOAT, value, allowEmpty);
    }
    
    /**
     * 是否是最长两位小数的金额，支持正负号前缀，如-12,-12.3,+12.35,12,12.3,12.35,0,0.3,0.35,-0.3,-0.35
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isAmount2R(String value, boolean allowEmpty)
    {
        return isRegexp(AMOUNT_2R, value, allowEmpty);
    }
    
    /**
     * 是否是邮箱地址，支持类似于xxx@xxx.xxx的邮箱格式，支持[-_.]等连接符，不支持中文等特殊符号
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isEmail(String value, boolean allowEmpty)
    {
        return isRegexp(EMAIL, value, allowEmpty);
    }
    
    /**
     * 是否是IP地址，支持类似于192.168.1.1/192.168.01.01/192.168.001.001的IP格式
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isIP(String value, boolean allowEmpty)
    {
        return isRegexp(IP, value, allowEmpty);
    }
    
    /**
     * 是否是MAC地址，格式为23:ad:87:be(十六进制)
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isMacAddress(String value, boolean allowEmpty)
    {
        return isRegexp(MAC, value, allowEmpty);
    }
    
    /**
     * 是否是泛手机号码，支持前缀+86,86开头,后面接手机号
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isMobile(String value, boolean allowEmpty)
    {
        return isRegexp(MOBILE, value, allowEmpty);
    }
    
    /**
     * 是否是中国区11位的手机号码
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isMobile11(String value, boolean allowEmpty)
    {
        return isRegexp(MOBILE_11, value, allowEmpty);
    }
    
    /**
     * 是否是日期格式，格式为yyyy-MM-dd
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isDate(String value, boolean allowEmpty)
    {
        return isRegexp(DATE, value, allowEmpty);
    }
    
    /**
     * 是否是时间格式，格式为HH:mm:ss
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isTime(String value, boolean allowEmpty)
    {
        return isRegexp(TIME, value, allowEmpty);
    }
    
    /**
     * 是否是日期时间格式，格式为yyyy-MM-dd HH:mm:ss
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isDateTime(String value, boolean allowEmpty)
    {
        return isRegexp(DATE_TIME, value, allowEmpty);
    }
    
    /**
     * 是否是身份证号，格式为[0-9]{6}yyyyMMdd[0-9]{3}[0-9xX*]，前6位数字，中间8位1900-2099年的日期，末4位为3个数字和[0-9]或xX*。
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isIdcard(String value, boolean allowEmpty)
    {
        return isRegexp(IDCARD, value, allowEmpty);
    }
    
    /**
     * 是否是邮政编码
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isPostalCode(String value, boolean allowEmpty)
    {
        return isRegexp(POSTAL_CODE, value, allowEmpty);
    }
    
    /**
     * 是否是合适的用户编码
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isUserCode(String value, boolean allowEmpty)
    {
        return isRegexp(USER_CODE, value, allowEmpty);
    }
    
    /**
     * 是否是合适的用户密码
     * 
     * @param value         需要验证的值
     * @param allowEmpty    是否允许为空
     * @return =true表示成功,=false表示失败
     */
    public static boolean isUserPass(String value, boolean allowEmpty)
    {
        return isRegexp(USER_PASS, value, allowEmpty);
    }
}
