/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_kernel.htm
 *
 * This file is part of [zhiqim_kernel].
 * 
 * [zhiqim_kernel] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_kernel] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_kernel].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.kernel.util;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;

import org.zhiqim.kernel.annotation.AnAlias;
import org.zhiqim.kernel.annotation.AnNullable;

/**
 * 操作系统命令相关工具类
 * 
 * @version v1.0.0 @author zouzhigang 2014-2-27 新建与整理
 */
@AnAlias("Runtimes")
public class Runtimes
{
    /***************************************************************/
    //读取命令操作的结果
    /***************************************************************/
    
    /**
     * 执行命令，无需返回值
     * 
     * @param command       无参数命令
     * @return              命令结果
     * @throws IOException  IO异常要求业务捕捉
     */
    public static void exec(String command) throws IOException
    {
        Runtime.getRuntime().exec(command);
    }
    
    /**
     * 无参数命令获取数据结果字符串
     * 
     * @param command       无参数命令
     * @param encoding      编码
     * @return              命令结果
     * @throws IOException  IO异常要求业务捕捉
     */
    public static String exec(String command, String encoding) throws IOException
    {
        return exec(new String[]{command, null, null}, encoding);
    }
    
    /**
     * 无参数命令获取数据结果字符串
     * 
     * @param cmdarray      命令集
     * @param encoding      编码
     * @return              命令结果
     * @throws IOException  IO异常要求业务捕捉
     */
    public static String exec(String command, Charset charset) throws IOException
    {
        return exec(new String[]{command, null, null}, charset);
    }
    
    /**
     * 有参数命令获取数据结果字符串
     * 
     * @param cmdarray      命令集
     * @param encoding      编码
     * @return              命令结果
     * @throws IOException  IO异常要求业务捕捉
     */
    public static String exec(String[] cmdarray, String encoding) throws IOException
    {
        return exec(cmdarray, Charset.forName(encoding));
    }
    
    /**
     * 有参数命令获取数据结果字符串
     * 
     * @param cmdarray      命令集
     * @param charset       字符集
     * @return              命令结果
     * @throws IOException  IO异常要求业务捕捉
     */
    public static String exec(String[] cmdarray, Charset charset) throws IOException
    {
        Process p =  Runtime.getRuntime().exec(cmdarray);
        return Streams.getString(p.getInputStream(), charset);
    }
    
    /**
     * 有参数命令获取数据结果行，指定需要的行数
     * 
     * @param cmd           命令行
     * @param encoding      编码
     * @param maxLineNum    行数，则读到指令行数即停止，0表示读所有
     * @return List<String> 字符串列表值
     * @throws IOException  IO异常要求业务捕捉
     */
    public static List<String> exec(String[] cmdarray, String encoding, int maxLineNum) throws IOException
    {
        return exec(cmdarray, Charset.forName(encoding), maxLineNum);
    }
    
    /**
     * 有参数命令获取数据结果行，指定需要的行数
     * 
     * @param cmd           命令行
     * @param charset       字符集
     * @param maxLineNum    行数，则读到指令行数即停止，0表示读所有
     * @return List<String> 字符串列表值
     * @throws IOException  IO异常要求业务捕捉
     */
    public static List<String> exec(String[] cmd, Charset charset, int maxLineNum) throws IOException
    {
        List<String> lines = new ArrayList<String>();
        InputStream is = null;
        BufferedReader reader = null;
        
        try
        {
            Process p =  Runtime.getRuntime().exec(cmd);
            reader = new BufferedReader(new InputStreamReader(p.getInputStream(), charset));
            
            String line = null;
            while((line = reader.readLine()) != null)
            {
                lines.add(line);
                
                if (maxLineNum > 0 && lines.size() >= maxLineNum)
                    return lines;
            }

            return lines;
        }
        finally
        {
            if (reader != null)
                try{reader.close();}catch(Exception e){}
                
            if (is != null)
                try{is.close();}catch(Exception e){}
        }
    }
    
    /***************************************************************/
    //双系统通用的功能
    /***************************************************************/
    
    /**
     * 获取第一个MAC地址，结果为null表示未找到
     * 
     * @param encoding      编码
     * @return              MAC地址
     * @throws IOException  IO异常
     */
    @AnNullable
    public static String getFirstMacAddress(String encoding) throws IOException
    {
        return Systems.isWindows()?Windows.getFirstMacAddress(encoding):Linux.getFirstMacAddress(encoding);
    }
}
