/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_kernel.htm
 *
 * This file is part of [zhiqim_kernel].
 * 
 * [zhiqim_kernel] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_kernel] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_kernel].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.kernel.util;

import java.text.DecimalFormat;

import org.zhiqim.kernel.annotation.AnAlias;
import org.zhiqim.kernel.constants.EnumConstants;

/**
 * 金额相关工具类
 *
 * @version v1.0.0 @author zouzhigang 2014-2-27 新建与整理
 */
@AnAlias("Amounts")
public class Amounts implements EnumConstants
{   
    /**
     * 金额分，转为字符串元，转化后是2位小数点
     * 
     * @param fen   金额分值，能转化成长整型的字符串
     * @return      金额元值字符串
     */
    public static String toYuan(String fen)
    {
        Asserts.asInteger(fen);
        return toYuan(Long.parseLong(fen));
    }
    
    /**
     * 金额分，转为字符串元，转化后最大2位小数点，没有的不保留小数点
     * 
     * @param fen   金额分值
     * @return      金额元值字符串
     */
    public static String toYuan(long fen)
    {
        int radixLen = 2;
        int divisor = 100;
        
        double result = (double)fen / divisor;
        StringBuilder radix = new StringBuilder("#");
        radix.append(".");
        for (int i=0;i<radixLen;i++)
            radix.append("#");
        
        DecimalFormat df = new DecimalFormat(radix.toString());
        return df.format(result);
    }
    
    /**
     * 金额分，转为字符串元，转化后强制为2位小数点，没有的为.00
     * 
     * @param fen   金额分值
     * @return      金额元值字符串
     */
    public static String toYuanMustRadix(String fen)
    {
        Asserts.asInteger(fen);
        return toYuanMustRadix(Long.parseLong(fen));
    }
    
    /**
     * 金额分，转为字符串元，转化后强制为2位小数点，没有的为.00
     * 
     * @param fen   金额分值
     * @return      金额元值字符串，如"2.00"
     */
    public static String toYuanMustRadix(long fen)
    {
        int radixLen = 2;
        int divisor = 100;
        
        double result = (double)fen / divisor;
        StringBuilder radix = new StringBuilder("#");
        radix.append(".");
        for (int i=0;i<radixLen;i++)
            radix.append("#");
        
        DecimalFormat df = new DecimalFormat(radix.toString());
        String ret = df.format(result);
        
        int ind = ret.indexOf('.');
        if (ind == -1)
        {//没有小数点
            ret += ".";
            for (int i=0;i<radixLen;i++)
                ret += "0";
        }
        else if (ind > ret.length() - radixLen -1)
        {//小数位数不足,尾部加0
            int zeroNum = ind - (ret.length() - radixLen - 1);
            for (int i=0;i<zeroNum;i++)
            {
                ret += "0";
            }
        }
        
        return ret;
    }
    
    /**
     * 金额元字符串转int金额分，支持整数、1位小数和2位小数点的金额字符串
     * 
     * @param yuan          金额元的字符串
     * @param defaultValue  缺省值
     * @return              金额分整型
     */
    public static int toFen(String yuan, int defaultValue)
    {
        if (yuan == null)
            return defaultValue;
        
        if (!Validates.isAmount2R(yuan))
            return defaultValue;
        
        boolean isNegative = false;
        if (yuan.startsWith("-"))
        {
            isNegative = true;
            yuan = yuan.substring(1);
        }
        
        int index = yuan.indexOf('.');
        if (index == -1)
        {
            int value = Integer.parseInt(yuan) * 100;//由元转为分
            return (isNegative)?-value:value;
        }
        
        int integer = Integer.parseInt(yuan.substring(0, index)) * 100;
        String radixString = yuan.substring(index + 1);
        if (radixString.length() == 0)
        {
            return (isNegative)?-integer:integer;
        }
        else if (radixString.length() == 1)
        {
            int radix = Integer.parseInt(radixString);
            int value = integer + radix * 10;
            return (isNegative)?-value:value;
        }
        else
        {
            int radix = Integer.parseInt(radixString);
            int value = integer + radix;
            return (isNegative)?-value:value;
        }
    }
    
    /**
     * 金额元字符串转long金额分，支持整数、1位小数和2位小数点的金额字符串
     * 
     * @param yuan          金额元的字符串
     * @param defaultValue  缺省值
     * @return              金额分的长整型
     */
    public static long toFen(String yuan, long defaultValue)
    {
        if (yuan == null)
            return defaultValue;
        
        if (!Validates.isAmount2R(yuan))
            return defaultValue;
        
        boolean isNegative = false;
        if (yuan.startsWith("-"))
        {
            isNegative = true;
            yuan = yuan.substring(1);
        }
        
        int index = yuan.indexOf('.');
        if (index == -1)
        {
            long value = Long.parseLong(yuan) * 100;//由元转为分
            return (isNegative)?-value:value;
        }
        
        long integer = Long.parseLong(yuan.substring(0, index)) * 100;
        String radixString = yuan.substring(index + 1);
        if (radixString.length() == 0)
        {
            return (isNegative)?-integer:integer;
        }
        else if (radixString.length() == 1)
        {
            long radix = Long.parseLong(radixString);
            long value = integer + radix * 10;
            return (isNegative)?-value:value;
        }
        else
        {
            long radix = Long.parseLong(radixString);
            long value = integer + radix;
            return (isNegative)?-value:value;
        }
    }
    
    /** 
     * 获取中文大写金额值 
     * 
     * @param value 单位为分的金额值
     */
    public static String toChinese(int value) 
    {
        Integer in = new Integer(value);
        double src = in.doubleValue();
        src = src/100;
        
        StringBuilder sb = new StringBuilder();
        DecimalFormat df = new DecimalFormat("0.00");
        String srcText = df.format(src);
        String numText = srcText.substring(0, srcText.length() - 3);
        String decText = srcText.substring(srcText.length() - 2);

        int numlen = numText.length();
        for (int k = 0; k < numlen; k++) 
        {
            sb.append(_RMB_NUM_[Integer.parseInt(String.valueOf(srcText.charAt(k)))]);
            sb.append(_RMB_UNIT_[numlen - k - 1]);
        }
        if ("00".equals(decText))
        {
            sb.append("整");
        } 
        else
        {
            sb.append(_RMB_NUM_[Integer.parseInt(String.valueOf(decText.charAt(0)))]);
            sb.append(_RMB_DEC_[0]);
            sb.append(_RMB_NUM_[Integer.parseInt(String.valueOf(decText.charAt(1)))]);
            sb.append(_RMB_DEC_[1]);
        }
        String result = sb.toString();
        result = result.replace("零分", "");
        result = result.replace("零角", "零");
        result = result.replace("零仟", "零");
        result = result.replace("零佰", "零");
        result = result.replace("零拾", "零");
        result = result.replace("零圆", "圆");
        while (true)
        {
            String r = result.replace("零零", "零");
            if (r.equals(result))
            {
                break;
            } 
            else 
            {
                result = r;
            }
        }
        result = result.replace("零圆", "圆");
        result = result.replace("零万", "万");
        result = result.replace("零亿", "亿");
        result = result.replace("亿万", "亿");
        if(result.startsWith("圆"))
        {
            result="零"+result;
        }
        return result;
    }
    
    /**
     * 通过原价和促销价，求折扣
     * 
     * @param price     原价
     * @param priceNew  促销价
     * @return 返回减钱折扣值，如9.92折,10折
     */
    public static String toDiscount(String price, String priceNew)
    {
        long priceLong = toFen(price, 0);
        long priceNewLong = toFen(priceNew, 0);
        return toDiscount(priceLong, priceNewLong);
    }
    
    /**
     * 通过原价和促销价，求折扣
     * 
     * @param price     原价
     * @param priceNew  促销价
     * @return 返回减钱折扣值，如9.92折,10折
     */
    public static String toDiscount(long price, long priceNew)
    {
        long priceSave = price - priceNew;
        if (priceSave == 0)
            return "10";
        else
            return toYuan(priceNew * 1000 / price);
    }
}
