/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_kernel.htm
 *
 * This file is part of [zhiqim_kernel].
 * 
 * [zhiqim_kernel] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_kernel] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_kernel].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.kernel.schedule;

import org.zhiqim.kernel.annotation.AnAlias;
import org.zhiqim.kernel.util.DateTimes;
import org.zhiqim.kernel.util.Longs;
import org.zhiqim.kernel.util.Strings;

/**
 * 每星期任务，设置星期、时、分、秒后在该点运行
 * 时间格式：yyyyMMdd，这里有点特殊的是每次计算上次日期
 * 
 * @version v1.0.0 @author zouzhigang 2014-2-27 新建与整理
 */
@AnAlias("Week")
public final class Week extends TaskThreader
{
    private int taskWeek    = 1;//指定处理的星期(1-7)
    private int taskHour    = 0;//指定处理的小时
    private int taskMinute  = 0;//指定处理的分钟
    private int taskSecond  = 0;//指定处理的秒
    
    /** 空构造函数,用于newInstance后再设置参数 */
    public Week(){}
    
    /**
     * 设置定时安排时间点，指定每月的时该任务运行
     * 
     * @param task      任务
     * @param week      指定周期，1-7表示星期一至星期七
     * @param hour      指定小时
     * @param minute    指定分钟
     * @param second    指定秒
     */
    public Week(Task task, int week, int hour, int minute, int second)
    {
        schedule(task, new int[]{week, hour, minute, second});
    }
    
    /**
     * 任务名称
     * 
     * @return          线程名称
     */
    public String getName()
    {
        return "Week";
    }
    
    /**
     * 任务时间安排
     * 
     * @return          时间安排
     */
    public String getTime()
    {
        return new StringBuilder().append(taskWeek).append("/").append(taskHour).append("/").append(taskMinute).append("/").append(taskSecond).toString();
    }
    
    /**
     * 时钟安排
     * 
     * @param task      任务
     * @param time      时间数组格式
     * @return          返回本对象
     */
    protected TaskThreader schedule(Task task, int[] time)
    {
        if (task == null || time == null || time.length != 4)
            throw new RuntimeException("Week传入的参数为空或数组长度不对");
        
        if (time[0] < 0 || time[0] > 7 || time[1] < 0 || time[1] > 23 || time[2] < 0 || time[2] > 59 || time[3] < 0 || time[3] > 59)
            throw new RuntimeException("Week传时的时间格式不正确");
        
        super.task = task;
        this.taskWeek = time[0];
        this.taskHour = time[1];
        this.taskMinute = time[2];
        this.taskSecond = time[3];
        return this;
    }
    
    @Override
    protected long calcLastTime(int curYearMonth, int curMonthMaxDay, int curDay, int curHour, int curMinute, int curSecond, int curWeek)
    {
        int weekDiffDays = curWeek - taskWeek;
        if (weekDiffDays == 0)
        {//当天
            return curYearMonth * 100L + curDay;
        }
        
        if (weekDiffDays < 0)
        {//偏移到下周了
            weekDiffDays += 7;
        }
        
        String curDate = (curYearMonth / 100) + "-" + Strings.prefixZero(curYearMonth % 100, 2) + "-" + Strings.prefixZero(curDay, 2);
        String lastWeekDate = DateTimes.getPreviousDateString(curDate, weekDiffDays);
        return Longs.toLong(lastWeekDate.replaceAll("-", ""));
    }
    
    @Override
    protected long calcLastTimeNextTime()
    {//7天后
        String lastWeekDate = (lastTime / 10000) + "-" + Strings.prefixZero((lastTime / 100) % 100, 2) + "-" + Strings.prefixZero(lastTime % 100, 2);
        String nextWeekDate = DateTimes.getNextDateString(lastWeekDate, 7);
        return Longs.toLong(nextWeekDate.replaceAll("-", ""));
    }
    
    @Override
    protected final boolean isArrive(int curYearMonth, int curMonthMaxDay, int curDay, int curHour, int curMinute, int curSecond, int curWeek)
    {
        //下周，星期相等，小时未达
        if (curHour < taskHour)
            return false;
        
        //下周，星期相等，小时已达
        if (curHour > taskHour)
            return true;
        
        //下周，星期小时相等，分钟未达
        if (curMinute < taskMinute)
            return false;
        
        //下周，星期小时相等，分钟已达
        if (curMinute > taskMinute)
            return true;
        
        //下周，星期小时分钟相等，比较秒数
        return curSecond >= taskSecond;
    }
    
    public int getTaskWeek()
    {
        return taskWeek;
    }

    public int getTaskHour()
    {
        return taskHour;
    }

    public int getTaskMinute()
    {
        return taskMinute;
    }

    public int getTaskSecond()
    {
        return taskSecond;
    }
}
