/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_kernel.htm
 *
 * This file is part of [zhiqim_kernel].
 * 
 * [zhiqim_kernel] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_kernel] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_kernel].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.kernel.schedule;

import org.zhiqim.kernel.annotation.AnAlias;
import org.zhiqim.kernel.constants.SignConstants;
import org.zhiqim.kernel.control.Threadx;
import org.zhiqim.kernel.logging.Log;
import org.zhiqim.kernel.logging.LogFactory;
import org.zhiqim.kernel.util.Threads;

/**
 * 使用线程实现一个单一的定时任务 <br><br>
 * 
 * 注意:该Interval的线程处理完后再sleep <br><br>
 * 该类的使用场景为：
 *      1）定时触发要求不严格 <br>
 *      2）不作时间追赶,不与系统时间比较 <br>
 *      3）该类的指Task类名为Timer线程名 <br>
 *      4）该类仅支持一个Task <br>
 *      
 * @version v1.0.0 @author zouzhigang 2014-2-27 新建与整理
 */
@AnAlias("Interval")
public class Interval extends Threadx implements SignConstants
{
    private static final Log log = LogFactory.getLog(Interval.class);
    
    private Task task;
    private int firstMillis;
    private int middleMillis;
    private int times;
    
    /********************************************************************************/
    //提交任务的定时安排 静态方法
    /********************************************************************************/
    
    /**
     * 创建Interval，支持传入任务，每次等待时间，启动立即运行，默认一直运行
     * 
     * @param task              任务
     * @param millis            每次等待时间毫秒数
     */
    public static Interval shedule(Task task, int millis)
    {
        return new Interval().open(task, 0, millis, INFINITE);
    }
    
    /**
     * 创建Interval，支持传入任务，时间参数，默认一直运行
     * 
     * @param task              任务
     * @param firstMillis       第一次等待时间毫秒数
     * @param middleMillis      每次等待时间毫秒数
     */
    public static Interval shedule(Task task, int firstMillis, int middleMillis)
    {
        return new Interval().open(task, firstMillis, middleMillis, INFINITE);
    }
    
    /**
     * 创建Interval，支持传入任务，时间参数和次数
     * 
     * @param task              任务
     * @param firstMillis       第一次等待时间毫秒数
     * @param middleMillis      每次等待时间毫秒数
     * @param times             执行次数
     */
    public static Interval shedule(Task task, int firstMillis, int middleMillis, int times)
    {
        return new Interval().open(task, firstMillis, middleMillis, times);
    }
    
    /********************************************************************************/
    //提交任务的定时安排 静态方法
    /********************************************************************************/
    
    /** 设置任务 */
    public Interval open(Task task, int firstMillis, int middleMillis, int times)
    {
        this.task = task;
        this.firstMillis = firstMillis;
        this.middleMillis = middleMillis;
        this.times = times;
        
        super.open();
        return this;
    }
    
    /*****************************************************************/
    //线程名和关闭操作
    /*****************************************************************/
    
    @Override /** 线程名 */
    public String getThreadName()
    {
        return _INTERVAL_ + "-" + task.getClass().getName();
    }
    
    @Override /** 线程关闭后 */
    public void closeAfter()
    {
        this.times = 0;
    }
    
    /*****************************************************************/
    //任务操作
    /*****************************************************************/
    
    @Override /** 线程首次运行 */
    protected void first()
    {
        if (task instanceof TaskLoad)
        {//有加载要求的任务，前置加载
            ((TaskLoad)task).load();
        }
        
        //首次等待时长
        Threads.sleepIgnoreException(firstMillis);
    }
    
    @Override /** 线程持续运行 */
    protected void loop()
    {
        if (times == 0)
        {//执行次数到达
            close();
            return;
        }
        
        try
        {
            task.execute();
        }
        catch(Throwable e)
        {
            if (times == INFINITE)
                log.error("Interval:[%s],在进行时Exception异常", e, getThreadName());
            else
                log.error("Interval:[%s],在进行%s时Exception异常", e, getThreadName(), ((times == INFINITE)?"":"第"+times+"次"));
        }
        
        if (times > 0)
            times--;
        
        Threads.sleepIgnoreException(middleMillis);
    }
    
    @Override
    public String toString()
    {
        return new StringBuilder("{name:Interval, ")
            .append("firstMillis:").append(firstMillis).append(", ")
            .append("middleMillis:").append(middleMillis).append(", ")
            .append("times:").append(times).append("}")
            .toString();
    }
}
