/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_kernel.htm
 *
 * This file is part of [zhiqim_kernel].
 * 
 * [zhiqim_kernel] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_kernel] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_kernel].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.kernel.logging.logger;

import java.io.File;

import org.zhiqim.kernel.logging.Log;
import org.zhiqim.kernel.util.DateTimes;

/**
 * 日期滚动文件日志器，根据日期切换日志文件，格式如info.log.2014-02-28,info.log..2014-03-01,...
 *
 * @version v1.0.0 @author zouzhigang 2014-2-27 新建与整理
 */
public class FileDailyLogger extends FileLogger
{
    private String fileDate;                //文件日志对应的日期
    private Object lock = new Object();     //文件锁
    
    /**
     * 日志记录器记录日志
     * 
     * @param log       日志对象
     * @param level     日志级别
     * @param message   日志消息
     * @param throwable 日志异常
     */
    public void logger(Log log, int level, String message, Throwable throwable)
    {
        if (level < super.getLevelMin() || level > super.getLevelMax())
            return;
        
        String curDate = DateTimes.getDateString();
        if (!curDate.equals(fileDate))
            convertWriter(curDate);
        
        print(level, message, throwable);
    }
    
    @Override
    public void setFilePath(String filePath)
    {
        super.setFilePath(filePath);
        convertWriter(DateTimes.getDateString());
    }
    
    /** 变更文件输出器 */
    private void convertWriter(String date)
    {
        synchronized (lock)
        {
            //再次判断，因为同步之前可能并发到达，第一个处理了，后面的无需处理
            if (date.equals(fileDate))
                return;
            
            super.closeWriter();                        //关闭可能的输出器
            super.setFilePathOnly(newFilePath(date));   //设置当前文件路径为加日期的文件路径
            super.createWriter();                       //创建当前输出器
            
            this.removeMaxFile();                       //删除多余的日志文件
            this.fileDate = date;                       //最后设置日期为当前日期
        }
    }
    
    /** 删除多余的日志文件，最多保存fileMaxNum天数 */
    private void removeMaxFile()
    {
        int fileMaxNum = super.getFileMaxNum();
        if (fileMaxNum <= 0)
            return;
        
        //把fileMaxNum之前的文件删除，一般一天一次，也只是删一个文件，除非修改配置文件
        String maxIndexDate = DateTimes.getPreviousDateString(fileMaxNum);
        
        String fileName = super.getFileName();
        String fileDir = super.getFileDir();
        File dir = new File(fileDir);
        for (File file : dir.listFiles()) 
        {
            if (!file.isFile() || file.getName().length() < 10 || !file.getName().endsWith("." + fileName))
                continue;
            
            String date = file.getName().substring(0, 10);
            if (date.compareTo(maxIndexDate) < 0)
                file.delete();
        }
    }
    
    private String newFilePath(String date)
    {
        return this.fileDir + "/" + date + "." + this.fileName;
    }
}
