/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_kernel.htm
 *
 * This file is part of [zhiqim_kernel].
 * 
 * [zhiqim_kernel] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_kernel] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_kernel].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.kernel.logging;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;

import org.zhiqim.kernel.Global;
import org.zhiqim.kernel.Servicer;
import org.zhiqim.kernel.annotation.AnNullable;
import org.zhiqim.kernel.config.Group;
import org.zhiqim.kernel.config.Item;
import org.zhiqim.kernel.extend.HashMapSV;
import org.zhiqim.kernel.logging.logger.ConsoleLogger;
import org.zhiqim.kernel.logging.logger.FileDailyLogger;
import org.zhiqim.kernel.logging.logger.FileLogger;
import org.zhiqim.kernel.util.Arrays;
import org.zhiqim.kernel.util.Strings;
import org.zhiqim.kernel.util.Validates;

/**
 * 标准日志服务
 *
 * @version v1.0.0 @author zouzhigang 2014-2-27 新建与整理
 */
public class LogServer extends Servicer implements LogService
{
    private static final HashMapSV<LogConfig> configMap = new HashMapSV<>();
    private static final HashMap<Integer, ArrayList<Logger>> rootMap = new HashMap<>();
    
    @Override
    public boolean create() throws Exception
    {
        if (!initLogConfig())
            return false;
        
        initRootAppender();
        LogFactory.updateLogService(this);
        return true;
    }
    
    @Override
    public void destroy() throws Exception
    {
        configMap.clear();
    }
    
    /** 获取配置列表 */
    public Collection<LogConfig> getConfigList()
    {
        return configMap.values();
    }
    
    /** 判断是否有日志器 */
    @Override
    public boolean isLoggerEnabled(Log log, int level)
    {
        List<Logger> loggerList = getLoggerList(log, level);
        return loggerList != null && !loggerList.isEmpty();
    }
    
    /** 获取对应的日志器列表 */
    @Override @AnNullable
    public List<Logger> getLoggerList(Log log, int level)
    {
        //先根据MustName查，存在配置则取配置要求
        List<Logger> loggerList = hasMustName(log, level);
        if (loggerList != null)
            return loggerList;
        
        //再根据MustClass查，存在配置则取配置要求
        loggerList = hasMustClass(log, level);
        if (loggerList != null)
            return loggerList;
        
        //未查到取根配置
        return rootMap.get(level);
    }
    
    /** 实始化日志配置 */
    private boolean initLogConfig()throws Exception
    {
        if (!Global.hasGroup(id))
        {
            System.out.println("日志配置["+id+"]不存在");
            return false;
        }
        
        //查出配置列表
        for (Item item : Global.getGroup(id).list())
        {
            String key = item.getKey();
            if (!Global.hasGroup(key))
            {
                System.out.println("日志配置["+key+"]不存在");
                return false;
            }
            
            String value = item.getString();
            if (Validates.isEmpty(value) || value.indexOf("-") == -1)
            {
                System.out.println("日志配置["+key+"]的value(级别)不正确，格式如debug-info");
                return false;
            }
            
            String[] levels = Arrays.toStringArray(value, "-");
            int levelMin = LogFactory.getLevel(levels[0]);
            int levelMax = LogFactory.getLevel(levels[1]);
            if (levelMin == 0 || levelMax == 0 || levelMax < levelMin)
            {
                System.out.println("日志配置["+key+"]级别配置有误，最小和最大值必须在(debug|info|error|warn|fatal)范围内，如error-fatal");
                return false;
            }
            
            //缺省文件日志路径如key=logging.info，则filePath=./logs/info.log
            String filePathDefault = "./logs/"+ Strings.trimLeft(key, id+".") +".log";
            
            Group group = Global.getGroup(key);
            LogConfig config = new LogConfig();
            config.setId(key);
            config.setLevelMin(levelMin);
            config.setLevelMax(levelMax);
            config.setConsoleOut(group.isTrue(CONSOLE_OUT));
            config.setConsolePattern(group.getString(CONSOLE_PATTERN, PATTERN_DEFAULT));
            config.setFileOut(group.isTrue(FILE_OUT));
            config.setFilePath(group.getString(FILE_PATH, filePathDefault));
            config.setFilePattern(group.getString(FILE_PATTERN, PATTERN_DEFAULT));
            config.setFileEncoding(group.getString(FILE_ENCODING, _UTF_8_));
            config.setFileMaxNum(group.getInt(FILE_MAX_NUM, 7));
            config.setFileMaxSize(group.getInt(FILE_MAX_SIZE, 100));
            config.setMustName(group.getString(MUST_NAME));
            
            String mustClass = group.getString(MUST_CLASS);
            if (Validates.isNotEmptyBlank(mustClass))
            {
                Class<?> mustClazz = Global.forName(mustClass);
                config.setMustClass(mustClazz);
            }
            
            if (config.isConsoleOut())
            {
                ConsoleLogger cLogger = new ConsoleLogger(config.getConsolePattern(), levelMin, levelMax);
                config.setConsoleLogger(cLogger);
            }
            
            if (config.isFileOut())
            {
                String loggerClass = group.getString(FILE_LOGGER);
                
                FileLogger logger = null;
                if (Validates.isEmpty(loggerClass))
                    logger = new FileDailyLogger();
                else
                {
                    Class<?> loggerClazz = Global.forName(loggerClass);
                    Object obj = loggerClazz.newInstance();
                    if (!(obj instanceof FileLogger))
                    {
                        System.out.println("日志配置["+group.getId()+"]file.logger未实现Logger接口");
                        return false; 
                    }
                    
                    logger = (FileLogger)obj;
                }
                
                logger.setLevelMin(levelMin);
                logger.setLevelMax(levelMax);
                logger.setFilePattern(config.getFilePattern());
                logger.setFileEncoding(config.getFileEncoding());
                logger.setFileMaxNum(config.getFileMaxNum());
                logger.setFileMaxSize(config.getFileMaxSize());
                logger.setFilePath(config.getFilePath());
                
                config.setFileLogger(logger);
            }
            
            configMap.put(group.getId(), config);
        }
        
        return true;
    }
    
    /** 初始化根配置 */
    private void initRootAppender()
    {
        List<LogConfig> rootList = new ArrayList<LogConfig>();
        Collection<LogConfig> configList = configMap.values();
        for (LogConfig config : configList)
        {
            if (!Validates.isEmpty(config.getMustName()) || !Validates.isEmpty(config.getMustClass()))
                continue;
            
            rootList.add(config);
        }
        
        //找到根配置
        for (LogConfig root : rootList)
        {
            int min = root.getLevelMin();
            int max = root.getLevelMax();
            
            for (int i=min;i<=max;i++)
            {
                ArrayList<Logger> loggerList = rootMap.get(i);
                if (loggerList == null)
                {
                    loggerList = new ArrayList<Logger>();
                    rootMap.put(i, loggerList);
                }
                
                if (root.isConsoleOut())
                    loggerList.add(root.getConsoleLogger());
                if (root.isFileOut())
                    loggerList.add(root.getFileLogger());
                
                //清除未使用的占位
                loggerList.trimToSize();
            }
        }
    }
    
    /** 读取有配置名称的日志器列表 */
    private List<Logger> hasMustName(Log log, int level)
    {
        if (Validates.isEmpty(log.getMustName()))
            return null;
            
        List<Logger> loggerList = null;
        for (LogConfig config : configMap.values())
        {
            if (Validates.isEmpty(config.getMustName()))
                continue;
            
            if (!config.getMustName().equals(log.getMustName()))
                continue;
            
            //只要有一个名称相符，则表示只能在配置中查找，不能递传到root
            if (loggerList == null)
                loggerList = new ArrayList<>();
            
            //如果等级未配置则表示不处理
            if (level < config.getLevelMin() || level > config.getLevelMax())
                continue;
            
            //找到一个合适的则返回该配置对应的Logger
            if (config.isConsoleOut())
                loggerList.add(config.getConsoleLogger());
            if (config.isFileOut())
                loggerList.add(config.getFileLogger());
        }
        
        return loggerList;
    }
    
    /** 读取有配置类的日志器列表 */
    private List<Logger> hasMustClass(Log log, int level)
    {
        if (log.getMustClass() == null)
            return null;
        
        List<Logger> loggerList = null;
        for (LogConfig config : configMap.values())
        {
            if (config.getMustClass() == null)
                continue;
            
            if (log.getMustClass() != config.getMustClass())
                continue;
            
            //只要有一个类名相符，则表示只能在配置中查找
            if (loggerList == null)
                loggerList = new ArrayList<Logger>();
            
            //如果等级未配置则表示不处理
            if (level < config.getLevelMin() || level > config.getLevelMax())
                continue;
            
            //找到一个合适的则返回该配置对应的Logger
            if (config.isConsoleOut())
                loggerList.add(config.getConsoleLogger());
            if (config.isFileOut())
                loggerList.add(config.getFileLogger());
        }
        
        return loggerList;
    }
}
