/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_kernel.htm
 *
 * This file is part of [zhiqim_kernel].
 * 
 * [zhiqim_kernel] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_kernel] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_kernel].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.kernel.control;

import org.zhiqim.kernel.Global;
import org.zhiqim.kernel.config.Group;
import org.zhiqim.kernel.config.Item;
import org.zhiqim.kernel.extend.HashMapCV;
import org.zhiqim.kernel.extend.MapCV;
import org.zhiqim.kernel.logging.Log;
import org.zhiqim.kernel.logging.LogFactory;
import org.zhiqim.kernel.util.Randoms;

/**
 * 对象处理线程池控制器，指定池最小值、最大值、初始值，允许线程空闲时长和允许缓存对象大小<br><br>
 * 1.创建时创建初始值大小的线程池，创建对象队列，和主线程<br>
 * 2.其他线程通过post传递对象可执行对象<br>
 * 3.主线程调度线程池处理对象，并检查线程池空闲线程
 *
 * @version v1.0.0 @author zouzhigang 2014-2-27 新建与整理
 */
public class HandlerController extends ThreaderController
{
    private static final Log log = LogFactory.getLog(HandlerController.class);
    private static final String DEFAULT = "default";
    
    private Handler<?> defaultHandler = null;          //缺省处理器
    private final MapCV<Handler<?>> handlerMap = new HashMapCV<>();
    
    public HandlerController()
    {
        setThreadName("Handler-" + Randoms.upperLetters(1)+Randoms.lowerLettersDigits(3));
    }
    
    /*************************************************************************************************/
    //创建&销毁，支持通过配置key=class,value=handler方式
    /*************************************************************************************************/
    
    @Override
    public boolean openBefore()
    {
        super.openBefore();
        
        if (id == null)
        {//没有配置服务
            return true;
        }
        
        Group group = Global.getGroup(id+".handler");
        if (group == null || group.isEmpty())
        {//没有配置处理器
            return true;
        }

        for (Item item : group.list())
        {
            String className = item.getKey();
            String handlerName = item.getString();

            Object handler = Global.newInstance(handlerName);
            if (handler == null || !(handler instanceof Handler))
            {
                log.error("配置[%s]中，[%s]类不存在或未实现Handler接口", group.getId(), handlerName);
                return false;
            }
            
            if (DEFAULT.equals(className))
            {//默认类名
                setDefaultHandler((Handler<?>)handler);
            }
            else
            {//指定类名
                Class<?> cls = Global.forName(className);
                if (cls == null)
                {
                    log.error("配置[%s]中，[%s]类不存在", group.getId(), className);
                    return false;
                }
                handlerMap.put(cls, (Handler<?>)handler);
            }
        }
        
        return true;
    }
    
    @Override
    public void closeAfter()
    {
        super.closeAfter();
        
        defaultHandler = null;
        handlerMap.clear();
    }
    
    /*************************************************************************************************/
    //增加&删除处理器
    /*************************************************************************************************/
    
    /** 设置缺省处理器 */
    public <T> void setDefaultHandler(Handler<T> handler)
    {
        defaultHandler = handler;
    }

    /** 清除缺省处理器 */
    public void clearDefaultHandler()
    {
        defaultHandler = null;
    }

    /** 增加一个对象处理器 */
    public <T> void addHandler(Class<T> requestClass, Handler<T> handler)
    {
        handlerMap.put(requestClass, handler);
    }

    /** 删除一个对象处理器 */
    public <T> void removeHandler(Class<T> requestClass)
    {
        handlerMap.remove(requestClass);
    }
    
    /*************************************************************************************************/
    //公开的对象处理方法
    /*************************************************************************************************/
    
    /** 处理一个事件，转化为Handlee由父类处理 */
    public boolean execute(Object event)
    {
        return execute(event, false);
    }
    
    /** 处理一个事件，并提供优先参数，转化为Handlee由父类处理 */
    public boolean execute(Object event, boolean priority)
    {
        Handler<?> handler = handlerMap.get(event.getClass());
        if (handler == null)
            handler = defaultHandler;
            
        if (handler == null)
        {
            log.fatal("事件处理器[%s]未配置处理器", event.getClass().getName());
            return false;
        }
        
        Handlee<?> command = new Handlee<>(event, handler);
        return super.execute(command, priority);
    }
}
