/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_kernel.htm
 *
 * This file is part of [zhiqim_kernel].
 * 
 * [zhiqim_kernel] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_kernel] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_kernel].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.kernel.config;

import java.io.Serializable;

import org.zhiqim.kernel.constants.ZhiqimConstants;
import org.zhiqim.kernel.constants.SignConstants;
import org.zhiqim.kernel.util.Ints;
import org.zhiqim.kernel.util.Longs;
import org.zhiqim.kernel.util.Validates;
import org.zhiqim.kernel.xml.Xmlx;

/**
 * 工程配置项，每个工程配置组有多个工程配置项<br><br>
 * 每个工程配置项，有key,value,type,desc四个属性<br>
 * key为键<br>
 * value为值<br>
 * type表示该值的类型private|protected|public，分别表示私有不展示给用户看，用户可见，用户可见可修改，默认protected<br>
 * desc表示该配置项的描述信息

 * 1.XML配置格式
 * <item key="name" value="Zhiqim Server" type="protected" desc="系统名称" />
 *  
 * 2.INI配置格式
 * #系统名称
 * name=Zhiqim Server
 * #private版本号,private表示type=private
 * version=v1.0.0
 * 
 * @version v1.0.0 @author zouzhigang 2014-2-27 新建与整理
 */
public class Item implements Serializable, ZhiqimConstants, SignConstants
{
    private static final long serialVersionUID = 1L;
    
    //父配置类
    private Group group;
    
    //配置参数
    private String key;
    private String value;
    private ItemType type;
    private String desc;
    private char symbol;
    
    public Item(Group group, String key, String value, ItemType type, String desc)
    {
        this(group, key, value, type, desc, '#');
    }
    
    public Item(Group group, String key, String value, ItemType type, String desc, char symbol)
    {
        this.group = group;
        this.key = key;
        this.value = group.isXml()?Xmlx.toXmlLoad(value):value;
        this.type = type;
        this.desc = desc;
        this.symbol = symbol;
    }
    
    public Item clone()
    {
        return new Item(group, key, value, type, desc, symbol);
    }
    
    /********************************************************************/
    //以下为Item运行时支持生成字符串和设置属性
    /********************************************************************/
    
    public String toString()
    {
        if (group.isXml())
        {
            return new StringBuilder("<item")
                .append(" key=").append(_DOUBLE_QUOTE_).append(key).append(_DOUBLE_QUOTE_)
                .append(" value=").append(_DOUBLE_QUOTE_).append(Xmlx.toXmlSave(value)).append(_DOUBLE_QUOTE_)
                .append(" type=").append(_DOUBLE_QUOTE_).append(ItemType.toString(type)).append(_DOUBLE_QUOTE_)
                .append(" desc=").append(_DOUBLE_QUOTE_).append(desc).append(_DOUBLE_QUOTE_)
                .append(" />")
                .toString();
        }
        else
        {
            StringBuilder strb = new StringBuilder();
            if (type != null || desc != null)
            {
                strb.append(symbol);
                if (type != null)
                    strb.append(type);
                if (desc != null)
                    strb.append(desc);
                strb.append(_BR_);
            }
            strb.append(key).append("=").append(value);
            return strb.toString();
        }
    }
    
    /** 设置值 */
    public Item setValue(String value)
    {
        this.value = value;
        return this;
    }
    
    /** 修改对应的组 */
    public Item setGroup(Group group)
    {
        this.group = group;
        return this;
    }
    
    /********************************************************************/
    //以下为Item属性
    /********************************************************************/

    public String getKey()
    {
        return key;
    }
    
    public char getSymbol()
    {
        return symbol;
    }
    
    public ItemType getType()
    {
        return type;
    }
    
    public String getTypeString()
    {
        return ItemType.toString(type);
    }
    
    public boolean isPublic()
    {
        return ItemType.PUBLIC == type;
    }
    
    public boolean isProtected()
    {
        return ItemType.PROTECTED == type;
    }
    
    public boolean isPrivate()
    {
        return ItemType.PRIVATE == type;
    }
    
    public String getDesc()
    {
        return desc;
    }

    public String getString()
    {
        return value;
    }
    
    public String getString(String defaultValue)
    {
        return Validates.isEmptyBlank(value)?defaultValue:value;
    }
    
    public String getStringFormat()
    {
        return Xmlx.toXmlSave(value);
    }
    
    public int getInt()
    {
        return Ints.toInt(value);
    }
    
    public int getInt(int defaultValue)
    {
        return Ints.toInt(value, defaultValue);
    }
    
    public long getLong()
    {
        return Longs.toLong(value);
    }
    
    public long getLong(long defaultValue)
    {
        return Longs.toLong(value, defaultValue);
    }
    
    public boolean isTrue()
    {
        return _TRUE_.equalsIgnoreCase(value);
    }
    

    /********************************************************************/
    //以下定义相等,当group和key都相等时才表示同一个
    /********************************************************************/

    @Override
    public int hashCode()
    {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((group == null) ? 0 : group.hashCode());
        result = prime * result + ((key == null) ? 0 : key.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj)
    {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        Item other = (Item) obj;
        if (group == null)
        {
            if (other.group != null)
                return false;
        }
        else if (!group.equals(other.group))
            return false;
        if (key == null)
        {
            if (other.key != null)
                return false;
        }
        else if (!key.equals(other.key))
            return false;
        return true;
    }
    
}