/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_kernel.htm
 *
 * This file is part of [zhiqim_kernel].
 * 
 * [zhiqim_kernel] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_kernel] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_kernel].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.kernel;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

import org.zhiqim.kernel.constants.ZhiqimConstants;
import org.zhiqim.kernel.constants.SignConstants;
import org.zhiqim.kernel.util.Files;
import org.zhiqim.kernel.util.Lists;
import org.zhiqim.kernel.util.Systems;

/**
 * 知启蒙工程类路径，默认取当前目录下的lib目录
 * 
 * @version v1.0.0 @author zouzhigang 2014-2-27 新建与整理
 */
final class ZhiqimClasspath implements ZhiqimConstants, SignConstants
{
    private List<String> classpathList = new ArrayList<>();
    private ClassLoader parentLoader = null;
    private ZhiqimBootLoader classLoader = null;
    
    /** 获取全局ClassLoader */
    public ClassLoader getClassLoader()
    {
        return this.classLoader;
    }
    
    /** 初始化CLASSPATH */
    public void initClasspath(String libPaths) throws IOException
    {
        classpathList.clear();
        
        HashSet<String> alreadys = new HashSet<>();
        
        //1.加入lib目录
        File libFolder = new File(Z_LIB_FOLDER);
        addJarFolder(Z_LIB_FOLDER, libFolder);
        alreadys.add(libFolder.getCanonicalPath());
        
        //2.加入配置的boot.lib目录下*.jar和*.zip
        List<String> libPathList = Lists.toStringList(libPaths);
        for (String libPath : libPathList)
        {
            File libDir = new File(libPath);
            if (!Files.isDirectory(libDir))
                continue;
            
            String canonicalPath = libDir.getCanonicalPath();
            if (alreadys.contains(canonicalPath))
                continue;//排重
            
            addJarFolder(libPath, libDir);
            alreadys.add(canonicalPath);
        }

        //3.设置ClassPath和ClassLoader
        Systems.setClassPath(toString());
        classLoader = getClassLoaderNew();
        Thread.currentThread().setContextClassLoader(classLoader);
    }
    
    public void releaseClasspath()
    {
        Thread.currentThread().setContextClassLoader(parentLoader);
        try{classLoader.close();}catch (IOException e){}
    }
    
    /** 加载目录下jar,zip到列表 */
    private void addJarFolder(String libPath, File libDir) throws IOException
    {
        //2.1把目录下*.jar和*.zip加入到classpath中
        File[] jarFileArr = libDir.listFiles(new FileFilter()   
        {   
            public boolean accept(File dir)   
            {
                String name = dir.getName().toLowerCase();
                return name.endsWith(".jar") || name.endsWith(".zip");   
            }   
        });   

        for (int i=0;i<jarFileArr.length;i++)
        {
            addJarFile(jarFileArr[i]);
        }
    
        //2.2把目录加到虚拟机动态库目录，方便JNI调用的.dll,.so等库
        Systems.addLibraryPath(libPath);
    }
    
    /** 加入JAR文件到列表 */
    private boolean addJarFile(File jar) throws IOException
    {
        if (jar == null || !jar.exists()) 
            return false;

        String key = jar.getCanonicalPath();
        if (!classpathList.contains(key))
        {
            classpathList.add(key);
            return true;
        }
  
        return false;
    }

    
    /** 创建新的类加载器 */
    private ZhiqimBootLoader getClassLoaderNew() throws MalformedURLException
    {
        int size = classpathList.size();
        URL[] urls = new URL[size];
        for (int i=0;i<size;i++) 
        {
            urls[i] = new File(classpathList.get(i)).toURI().toURL();
        }
        
        parentLoader = Thread.currentThread().getContextClassLoader();
        if (parentLoader == null)
            parentLoader = ZhiqimClasspath.class.getClassLoader();

        if (parentLoader == null)
            parentLoader = ClassLoader.getSystemClassLoader();

        return new ZhiqimBootLoader(urls, parentLoader);
    }

    public String toString()
    {
        int size = classpathList.size();
        if (size == 0)
            return _EMPTY_;
        
        StringBuilder strb = new StringBuilder(1024);
        //第一个
        strb.append(classpathList.get(0));
        //余下的
        for (int i=1;i<size;i++) 
        {
            strb.append(File.pathSeparatorChar).append(classpathList.get(i));
        }
        
        return strb.toString();
    }

    /** 定义一个引导加载器 */
    private class ZhiqimBootLoader extends URLClassLoader
    {
        private String name = "ZhiqimBootLoader";
        
        private ZhiqimBootLoader(URL[] urls, ClassLoader parent)
        {
            super(urls, parent);
        }

        public String toString()
        {
            return name;
        }
    }
}
