/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[欢迎加盟知启蒙，一起邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_manager.htm
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.zhiqim.manager.presenter;

import java.util.List;

import org.zhiqim.httpd.HttpRequest;
import org.zhiqim.httpd.context.annotation.AnIntercept;
import org.zhiqim.httpd.validate.ones.IsNotEmpty;
import org.zhiqim.kernel.annotation.AnAlias;
import org.zhiqim.kernel.util.Arrays;
import org.zhiqim.kernel.util.Lists;
import org.zhiqim.kernel.util.Validates;
import org.zhiqim.manager.ZmrBootstrap;
import org.zhiqim.manager.ZmrSessionUser;
import org.zhiqim.manager.dao.ZmrOperatorDao;
import org.zhiqim.manager.dbo.ZmrDept;
import org.zhiqim.manager.dbo.ZmrOperator;
import org.zhiqim.manager.dbo.ZmrRole;
import org.zhiqim.orm.dbo.Updater;

/**
 * 操作员控制器
 *
 * @version v1.0.0 @author zouzhigang 2017-6-6 新建与整理
 */
@AnAlias("ZmrOperatorPresenter")
@AnIntercept("chkZmrLogin")
public class ZmrOperatorPresenter
{
    /**
     * 验证操作员密码
     * 
     * @param request       请求
     * @param operatorPass  密码
     */
    public static void doValidatePassword(HttpRequest request, String operatorPass)
    {
        if (Validates.isEmptyBlank(operatorPass))
        {
            request.setResponseError("密码不允许为空白");
            return;
        }
        
        ZmrSessionUser sessionUser = request.getSessionUser(ZmrSessionUser.class);
        if (!ZmrOperatorDao.validatePassword(request, sessionUser.getOperator(), operatorPass))
        {
            request.setResponseError("密码不正确");
        }
    }
    
    /**
     * 增加操作员多个部门，参数统一在request中
     * 
     * @param request       请求
     * @throws Exception    异常
     */
    public static void doAddOperatorDepts(HttpRequest request) throws Exception
    {
        request.addValidate(new IsNotEmpty("operatorCode", "请选择操作员"));
        request.addValidate(new IsNotEmpty("deptIds", "请选择一个部门"));
        if (!request.chkValidate())
        {
            request.setResponseError(request.getAlertMsg());
            return;
        }
        
        //判断操作员是否存在
        String operatorCode = request.getParameter("operatorCode");
        ZmrOperator operator = ZmrBootstrap.tableOpr(request).item(ZmrOperator.class, operatorCode);
        if (operator == null)
        {
            request.returnHistory("您选择的操作员不存在，请选择一个有效的操作员");
            return;
        }
        
        //判断部门是否存在
        long[] deptIdArr = Arrays.toLongArray(request.getParameter("deptIds"));
        for (long deptId : deptIdArr)
        {
            ZmrDept dept = ZmrBootstrap.table(request).item(ZmrDept.class, deptId);
            if (operator == null)
                continue;
        
            //操作员部门
            if (Validates.isEmpty(operator.getOperatorDept()))
                operator.setOperatorDept("" + deptId);
            else
                operator.setOperatorDept(operator.getOperatorDept() + "," + deptId);
            
            //操作员所有部门
            String operatorDeptAll = operator.getOperatorDeptAll();
            if (Validates.isEmpty(operatorDeptAll))
                operatorDeptAll = dept.getDeptParentAll();
            else
                operatorDeptAll += "," + dept.getDeptParentAll();
            operatorDeptAll += "," + operator.getOperatorDept();
            operator.setOperatorDeptAll(operatorDeptAll);
        }
        
        Updater updater = new Updater();
        updater.addField("operatorDept", operator.getOperatorDept());
        updater.addField("operatorDeptAll", Arrays.toFilterSameStr(operator.getOperatorDeptAll()));
        updater.addMust("operatorCode", operatorCode);
        
        ZmrBootstrap.tableOpr(request).update(ZmrOperator.class, updater);
    }
    
    /**
     * 删除操作员部门，参数统一在request中
     * 
     * @param request       请求
     * @throws Exception    异常
     */
    public static void doDeleteOperatorDept(HttpRequest request) throws Exception
    {
        request.addValidate(new IsNotEmpty("operatorCode", "请选择一个操作员"));
        request.addValidate(new IsNotEmpty("deptId", "请选择一个部门"));
        if (!request.chkValidate())
        {
            request.setResponseError(request.getAlertMsg());
            return;
        }
        
        String operatorCode = request.getParameter("operatorCode");
        ZmrOperator operator = ZmrBootstrap.tableOpr(request).item(ZmrOperator.class, operatorCode);
        if (operator == null)
        {
            request.setResponseError("请选择一个有效的操作员");
            return;
        }
        
        //删除对应的部门
        long deptId = request.getParameterLong("deptId");
        String operatorDept = operator.getOperatorDept();
        List<Long> operatorDeptList = Lists.toLongList(operatorDept);
        operatorDeptList.remove(deptId);
        
        operatorDept = Lists.toString(operatorDeptList);
        
        //重置所有部门
        String operatorDeptAll = null;
        for (long oprDeptId : operatorDeptList)
        {
            ZmrDept dept = ZmrBootstrap.table(request).item(ZmrDept.class, oprDeptId);
            if (dept == null)
                continue;
            
            if (operatorDeptAll == null)
                operatorDeptAll = dept.getDeptParentAll();
            else
                operatorDeptAll += "," + dept.getDeptParentAll();
            
            operatorDeptAll += "," + oprDeptId;
        }
        operatorDeptAll = Arrays.toFilterSameStr(operatorDeptAll);
        
        Updater updater = new Updater();
        updater.addField("operatorDept", operatorDept);
        updater.addField("operatorDeptAll", operatorDeptAll);
        updater.addMust("operatorCode", operatorCode);
        
        ZmrBootstrap.tableOpr(request).update(ZmrOperator.class, updater);
    }
    
    /**
     * 增加操作员多个角色，参数统一在request中
     * 
     * @param request       请求
     * @throws Exception    异常
     */
    public static void doAddOperatorRoles(HttpRequest request) throws Exception
    {
        request.addValidate(new IsNotEmpty("operatorCode", "请选择操作员"));
        request.addValidate(new IsNotEmpty("roleIds", "请选择一个角色"));
        if (!request.chkValidate())
        {
            request.setResponseError(request.getAlertMsg());
            return;
        }
        
        //判断操作员是否存在
        String operatorCode = request.getParameter("operatorCode");
        ZmrOperator operator = ZmrBootstrap.tableOpr(request).item(ZmrOperator.class, operatorCode);
        if (operator == null)
        {
            request.returnHistory("您选择的操作员不存在，请选择一个有效的操作员");
            return;
        }
        
        //判断角色是否存在
        String[] roleIdArr = Arrays.toStringArray(request.getParameter("roleIds"));
        for (String roleId : roleIdArr)
        {
            if (Validates.isEmpty(operator.getOperatorRole()))
                operator.setOperatorRole(roleId);
            else if (!operator.getOperatorRole().contains(roleId))
                operator.setOperatorRole(operator.getOperatorRole() + "," + roleId);
        }
        
        Updater updater = new Updater();
        updater.addField("operatorRole", operator.getOperatorRole());
        updater.addMust("operatorCode", operatorCode);
        
        ZmrBootstrap.tableOpr(request).update(ZmrOperator.class, updater);
    }
    
    /**
     * 删除操作员角色，参数统一在request中
     * 
     * @param request       请求
     * @throws Exception    异常
     */
    public static void doDeleteOperatorRole(HttpRequest request) throws Exception
    {
        request.addValidate(new IsNotEmpty("operatorCode", "请选择一个操作员"));
        request.addValidate(new IsNotEmpty("roleId", "请选择一个角色"));
        if (!request.chkValidate())
        {
            request.setResponseError(request.getAlertMsg());
            return;
        }
        
        String operatorCode = request.getParameter("operatorCode");
        ZmrOperator operator = ZmrBootstrap.tableOpr(request).item(ZmrOperator.class, operatorCode);
        if (operator == null)
        {
            request.setResponseError("请选择一个有效的操作员");
            return;
        }
        
        //删除对应的角色
        long roleId = request.getParameterLong("roleId");
        String operatorRole = operator.getOperatorRole();
        List<Long> operatorRoleList = Lists.toLongList(operatorRole);
        operatorRoleList.remove(roleId);
        
        operatorRole = Lists.toString(operatorRoleList);
        
        Updater updater = new Updater();
        updater.addField("operatorRole", operatorRole);
        updater.addMust("operatorCode", operatorCode);
        
        ZmrBootstrap.tableOpr(request).update(ZmrOperator.class, updater);
    }
    
    /** 增加角色多个操作员 */
    public static void doAddRoleOperator(HttpRequest request) throws Exception
    {
        request.addValidate(new IsNotEmpty("roleId", "请选择一个角色"));
        request.addValidate(new IsNotEmpty("operatorCodes", "请选择操作员"));
        if (!request.chkValidate())
        {
            request.setResponseError(request.getAlertMsg());
            return;
        }
        
        //判断角色是否存在
        long roleId = request.getParameterLong("roleId");
        ZmrRole role = ZmrBootstrap.table(request).item(ZmrRole.class, roleId);
        if (role == null)
        {
            request.returnHistory("您选择的角色不存在，请选择一个有效的角色");
            return;
        }
        
        String[] operatorCodeArr = Arrays.toStringArray(request.getParameter("operatorCodes"));
        for (String operatorCode : operatorCodeArr)
        {
            ZmrOperator operator = ZmrBootstrap.tableOpr(request).item(ZmrOperator.class, operatorCode);
            if (operator == null)
                continue;
        
            //操作员角色
            String operatorRole = operator.getOperatorRole();
            if (Validates.isEmpty(operatorRole))
                operatorRole = "" + roleId;
            else
                operatorRole += "," + roleId;
            
            Updater updater = new Updater();
            updater.addField("operatorRole", operatorRole);
            updater.addMust("operatorCode", operatorCode);
            
            ZmrBootstrap.tableOpr(request).update(ZmrOperator.class, updater);
        }
    }
    
    /** 剔除在线用户 */
    public static void RemoveOnlineOperator(HttpRequest request, String sessionId) throws Exception
    {
        request.getContext().getSessionManager().invalidateSession(sessionId);
    }
    
    /**
     * 设置操作员组织编号
     * 
     * @param request       请求对象
     * @param orgId     组织编号
     * @throws Exception    异常
     */
    public static void setOrgId(HttpRequest request, long orgId) throws Exception
    {
        ZmrSessionUser sessionUser = request.getSessionUser(ZmrSessionUser.class);
        sessionUser.setValue("orgId", orgId);
    }
}
