/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[欢迎加盟知启蒙，一起邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_manager.htm
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.zhiqim.manager.dao;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.zhiqim.httpd.HttpRequest;
import org.zhiqim.httpd.HttpSessionUser;
import org.zhiqim.kernel.annotation.AnAlias;
import org.zhiqim.kernel.json.Jsons;
import org.zhiqim.kernel.util.Asserts;
import org.zhiqim.kernel.util.Lists;
import org.zhiqim.kernel.util.Validates;
import org.zhiqim.manager.ZmrBootstrap;
import org.zhiqim.manager.ZmrPassworder;
import org.zhiqim.manager.ZmrSessionUser;
import org.zhiqim.manager.dbo.ZmrDeptRule;
import org.zhiqim.manager.dbo.ZmrMenu;
import org.zhiqim.manager.dbo.ZmrOperator;
import org.zhiqim.manager.dbo.ZmrOperatorRule;
import org.zhiqim.manager.dbo.ZmrOrgRule;
import org.zhiqim.manager.dbo.ZmrRoleRule;
import org.zhiqim.orm.dbo.Selector;
import org.zhiqim.orm.dbo.Updater;

/**
 * 操作员数据访问对象
 *
 * @version v1.0.0 @author zouzhigang 2017-8-12 新建与整理
 */
@AnAlias("ZmrOperatorDao")
public class ZmrOperatorDao
{
    /*********************************************************************************************/
    //操作员基础信息
    /*********************************************************************************************/
    
    /**
     * 验证操作员密码
     * 
     * @param request       请求对象
     * @param operator      操作员对象
     * @param operatorPass  操作员密码
     * @return              =true表示成功
     * @throws Exception    可能的异常
     */
    public static boolean validateSecret(HttpRequest request, ZmrOperator operator, String operatorPass)
    {
        Asserts.as(operator!=null?null:"操作员对象不能为null");
        Asserts.as(!Validates.isEmptyBlank(operatorPass)?null:"操作员密码不能为空白");
        
        ZmrPassworder passworder = request.getContextAttribute(ZmrBootstrap.class).getPassworder();
        String encodePass = passworder.encode(operator.getOperatorCode(), operatorPass, operator.getOperatorPassSalt());
        return encodePass.equals(operator.getOperatorPass());
    }
    
    /**
     * 验证操作员密码
     * 
     * @param request       请求对象
     * @param operator      操作员对象
     * @param operatorPass  操作员密码
     * @return              =true表示成功
     * @throws Exception    可能的异常
     */
    public static boolean validatePassword(HttpRequest request, ZmrOperator operator, String operatorPass)
    {
        Asserts.as(operator!=null?null:"操作员对象不能为null");
        Asserts.as(!Validates.isEmptyBlank(operatorPass)?null:"操作员密码不能为空白");
        
        ZmrPassworder passworder = request.getContextAttribute(ZmrBootstrap.class).getPassworder();
        String encodePass = passworder.encode(operator.getOperatorCode(), operatorPass, operator.getOperatorPassSalt());
        return encodePass.equalsIgnoreCase(operator.getOperatorPass());
    }
    
    /**
     * 验证操作员密码
     * 
     * @param request       请求对象
     * @param operatorCode  操作员编码
     * @param operatorPass  操作员密码
     * @return              =true表示成功
     * @throws Exception    可能的异常
     */
    public static boolean validatePassword(HttpRequest request, String operatorCode, String operatorPass) throws Exception
    {
        Asserts.as(!Validates.isEmptyBlank(operatorCode)?null:"操作员编码不能为空白");
        Asserts.as(!Validates.isEmptyBlank( operatorPass)?null:"操作员密码不能为空白");
        
        ZmrOperator operator = ZmrBootstrap.tableOpr(request).item(ZmrOperator.class, operatorCode);
        if (operator == null)
            return false;
        
        return validatePassword(request, operator, operatorPass);
    }
    
    /**
     * 验证操作员是否有指定的菜单权限
     * 
     * @param request       请求对象
     * @param operatorCode  操作员编码
     * @param menuUrl       菜单URL
     * @return              =true表示有权限
     * @throws Exception    可能的异常
     */
    public static boolean hasMenuRule(HttpRequest request, String operatorCode, String menuUrl) throws Exception
    {
        Asserts.as(!Validates.isEmptyBlank(operatorCode)?null:"操作员编码不能为空白");
        Asserts.as(!Validates.isEmptyBlank(menuUrl)?null:"菜单URL不能为空白");
        
        List<ZmrMenu> menuList = getOperatorMenuList(request, operatorCode);
        for (ZmrMenu menu : menuList)
        {
            if (menuUrl.equals(menu.getMenuUrl()))
                return true;
        }
        
        return false;
    }
    
    /**
     * 验证操作员是否有指定的菜单权限
     * 
     * @param request       请求对象
     * @param operator      操作员对象
     * @param menuUrl       菜单URL
     * @return              =true表示有权限
     * @throws Exception    可能的异常
     */
    public static boolean hasMenuRule(HttpRequest request, ZmrOperator operator, String menuUrl) throws Exception
    {
        Asserts.as(operator != null?null:"操作员编码不能为空白");
        Asserts.as(!Validates.isEmptyBlank(menuUrl)?null:"菜单URL不能为空白");
        
        List<ZmrMenu> menuList = getOperatorMenuList(request, operator);
        for (ZmrMenu menu : menuList)
        {
            if (menuUrl.equals(menu.getMenuUrl()))
                return true;
        }
        
        return false;
    }
    
    /*********************************************************************************************/
    //操作员参数相关
    /*********************************************************************************************/
    
    /**
     * 获取操作员参数
     * 
     * @param request       请求对象
     * @param operatorCode  操作员编码
     * @param key           参数字段
     * @throws Exception    可能的异常
     */
    public static String getOperatorParam(HttpRequest request, String operatorCode, String key) throws Exception
    {
        Asserts.as(!Validates.isEmptyBlank(operatorCode)?null:"操作员编码不能为空白");
        
        ZmrOperator operator = ZmrBootstrap.tableOpr(request).item(ZmrOperator.class, operatorCode);
        if (operator == null)
            return null;
        
        return Jsons.getString(operator.getOperatorParam(), key);
    }
    
    /**
     * 获取操作员参数
     * 
     * @param request       请求对象
     * @param operator      操作员对象
     * @param key           参数字段
     * @throws Exception    可能的异常
     */
    public static String getOperatorParam(ZmrOperator operator, String key) throws Exception
    {
        Asserts.as(operator != null?null:"操作员编码不能为空白");
        
        return Jsons.getString(operator.getOperatorParam(), key);
    }
    
    /**
     * 更新操作员参数
     * 
     * @param request       请求对象
     * @param operatorCode  操作员编码
     * @param key           参数字段
     * @param value         参数值
     * @throws Exception    可能的异常
     */
    public static void addOrUpdateOperatorParam(HttpRequest request, String operatorCode, String key, Object value) throws Exception
    {
        ZmrOperator operator = ZmrBootstrap.tableOpr(request).item(ZmrOperator.class, operatorCode);
        if (operator == null)
            return;
        
        String operatorParam = operator.getOperatorParam();
        if (Validates.isEmptyBlank(operatorParam))
            operatorParam = "{}";
        operatorParam = Jsons.toStringAddOrUpdate(operatorParam, key, value);
        
        Updater updater = new Updater();
        updater.addField("operatorParam", operatorParam);
        updater.addMust("operatorCode", operatorCode);
        ZmrBootstrap.tableOpr(request).update(ZmrOperator.class, updater);
    }
    
    /**
     * 删除操作员参数
     * 
     * @param request       请求对象
     * @param operatorCode  操作员编码
     * @param key           参数字段
     * @throws Exception    可能的异常
     */
    public static void deleteOperatorParam(HttpRequest request, String operatorCode, String key) throws Exception
    {
        ZmrOperator operator = ZmrBootstrap.tableOpr(request).item(ZmrOperator.class, operatorCode);
        if (operator == null)
            return;
        
        String operatorParam = operator.getOperatorParam();
        if (Validates.isEmptyBlank(operatorParam))
            operatorParam = "{}";
        operatorParam = Jsons.toStringRemove(operatorParam, key);
        
        Updater updater = new Updater();
        updater.addField("operatorParam", operatorParam);
        updater.addMust("operatorCode", operatorCode);
        ZmrBootstrap.tableOpr(request).update(ZmrOperator.class, updater);
    }
    
    /*********************************************************************************************/
    //操作员权限相关
    /*********************************************************************************************/
    
    /**
     * 获取所有正常菜单列表
     * 
     * @param request       请求对象
     * @return              全部菜单列表
     * @throws Exception    可能的异常
     */
    public static List<ZmrMenu> getAllMenuList(HttpRequest request) throws Exception
    {
        Selector selector = new Selector();
        selector.addOrderbyAsc("menuLevel");
        selector.addOrderbyAsc("menuCode");
        
        return ZmrBootstrap.table(request).list(ZmrMenu.class, selector);
    }
    
    /**
     * 获取操作员菜单列表
     * 
     * @param request       请求对象
     * @param operatorCode  操作员编码
     * @return              菜单列表
     * @throws Exception    可能的异常
     */
    public static List<ZmrMenu> getOperatorMenuList(HttpRequest request, String operatorCode) throws Exception
    {
        Asserts.assertNotEmpty(operatorCode, "操作员编码不允许为空");
        
        ZmrOperator operator = ZmrBootstrap.tableOpr(request).item(ZmrOperator.class, operatorCode);
        if (operator == null)
            return new ArrayList<ZmrMenu>();
        
        return getOperatorMenuList(request, operator);
    }
    
    /**
     * 获取操作员菜单列表
     * 
     * @param request       请求对象
     * @param operator      操作员
     * @return              菜单列表
     * @throws Exception    可能的异常
     */
    public static List<ZmrMenu> getOperatorMenuList(HttpRequest request, ZmrOperator operator) throws Exception
    {
        List<ZmrMenu> menuList = getAllMenuList(request);
        if (operator.getOperatorType() == 0)
            return menuList;
        
        //1.删除未开启的功能菜单
        for (Iterator<ZmrMenu> it=menuList.iterator();it.hasNext();)
        {
            ZmrMenu menu = it.next();
            if (menu.getMenuStatus() != 0)
                it.remove();
        }
        
        //2.查出操作员独立权限、组织权限、部门权限和角色权限
        List<ZmrOperatorRule> operatorRuleList = ZmrBootstrap.table(request).list(ZmrOperatorRule.class, new Selector("operatorCode", operator.getOperatorCode()));
        
        //查看该操作员是否是所属组织负责人，如果是，给操作员赋予组织全部权限。
        List<ZmrOrgRule> orgRuleList = new ArrayList<ZmrOrgRule>();
        if(ZmrOrgDao.isOrgManager(request, operator.getOperatorCode()))
        {
            List<ZmrOrgRule> list = ZmrBootstrap.table(request).list(ZmrOrgRule.class, new Selector("orgId", operator.getOrgId()));
            orgRuleList.addAll(list);
        }
        
        List<ZmrDeptRule> deptRuleList = new ArrayList<>();
        String deptIds = ZmrParamDao.isOperatorDeptAllRule(request.getContext())?operator.getOperatorDeptAll():operator.getOperatorDept();
        List<Long> deptIdList = Lists.toLongList(deptIds);
        for (long deptId : deptIdList)
        {
            List<ZmrDeptRule> ruleList = ZmrBootstrap.table(request).list(ZmrDeptRule.class, new Selector("deptId", deptId));
            deptRuleList.addAll(ruleList);
        }
        
        List<ZmrRoleRule> roleRuleList = new ArrayList<>();
        List<Long> roleIdList = Lists.toLongList(operator.getOperatorRole());
        for (long roleId : roleIdList)
        {
            List<ZmrRoleRule> ruleList = ZmrBootstrap.table(request).list(ZmrRoleRule.class, new Selector("roleId", roleId));
            roleRuleList.addAll(ruleList);
        }
        
        //3.属于三者之一则认为有效
        for (Iterator<ZmrMenu> it=menuList.iterator();it.hasNext();)
        {
            String menuCode = it.next().getMenuCode();
            if (isOrgRule(orgRuleList, menuCode))
                continue;
            
            if (isOperatorRule(operatorRuleList, menuCode))
                continue;
            
            if (isDeptRule(deptRuleList, menuCode))
                continue;
            
            if (isRoleRule(roleRuleList, menuCode))
                continue;

            it.remove();
        }
            
        return menuList;
    }
    
    /**
     * 根据组织编号查询操作员
     * 
     * @param request       请求对象
     * @param orgId         组织编号
     * @return              组织下操作员列表
     * @throws Exception    可能的异常
     */
    public static List<ZmrOperator> getOperator(HttpRequest request, long orgId) throws Exception
    {
        return ZmrBootstrap.tableOpr(request).list(ZmrOperator.class, new Selector("orgId",orgId));
    }
    
    /**
     * 查询所有操作员
     * 
     * @param request       请求对象
     * @return              操作员列表
     * @throws Exception    可能的异常
     */
    public static List<ZmrOperator> getOperatorAll(HttpRequest request) throws Exception
    {
        return ZmrBootstrap.tableOpr(request).list(ZmrOperator.class);
    }
    
    /**
     * 指定操作员编码查询操作员
     * 
     * @param request       请求对象
     * @param operatorCode  操作员编号
     * @return              操作员对象
     * @throws Exception    可能的异常
     */
    public static ZmrOperator getOperator(HttpRequest request, String operatorCode) throws Exception
    {
        return ZmrBootstrap.tableOpr(request).item(ZmrOperator.class, operatorCode);
    }
    
    /** 是否在操作员列表中 */
    public static boolean isOperator(List<ZmrOperator> list, String operatorCode)
    {
        for (ZmrOperator opr : list)
        {
            if (operatorCode.equals(opr.getOperatorCode()))
                return true;
        }
        
        return false;
    }
    
    /** 是否操作员独立权限 */
    public static boolean isOperatorRule(List<ZmrOperatorRule> ruleList, String menuCode)
    {
        for (ZmrOperatorRule rule : ruleList)
        {
            if (menuCode.equals(rule.getMenuCode()))
                return true;
        }
        
        return false;
    }
    
    /** 是否部门权限 */
    public static boolean isDeptRule(List<ZmrDeptRule> ruleList, String menuCode)
    {
        for (ZmrDeptRule rule : ruleList)
        {
            if (menuCode.equals(rule.getMenuCode()))
                return true;
        }
        
        return false;
    }
    
    /** 是否组织权限 */
    public static boolean isOrgRule(List<ZmrOrgRule> ruleList, String menuCode)
    {
        for (ZmrOrgRule rule : ruleList)
        {
            if (menuCode.equals(rule.getMenuCode()))
                return true;
        }
        
        return false;
    }
    
    /** 是否角色权限 */
    public static boolean isRoleRule(List<ZmrRoleRule> ruleList, String menuCode)
    {
        for (ZmrRoleRule rule : ruleList)
        {
            if (menuCode.equals(rule.getMenuCode()))
                return true;
        }
        
        return false;
    }
    
    /** 刷新所有在线用户会话 */
    public static void refresh(HttpRequest request) throws Exception
    {
        List<HttpSessionUser> sessionUserList = request.getContext().getSessionManager().getSessionUserList();
        for (HttpSessionUser sessionUser : sessionUserList)
        {
            if (!(sessionUser instanceof ZmrSessionUser))
                continue;
            
            ZmrSessionUser sUser = (ZmrSessionUser)sessionUser;
            List<ZmrMenu> menuList = ZmrOperatorDao.getOperatorMenuList(request, sUser.getOperator());
            sUser.setMenuList(request.getContext(), menuList);
        }
    }
}
