/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[欢迎加盟知启蒙，一起邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_manager.htm
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.zhiqim.manager;

import java.util.List;

import org.zhiqim.httpd.HttpContext;
import org.zhiqim.httpd.HttpRequest;
import org.zhiqim.kernel.Global;
import org.zhiqim.kernel.config.ItemType;
import org.zhiqim.kernel.extend.SS;
import org.zhiqim.kernel.schedule.Day;
import org.zhiqim.kernel.schedule.Scheduler;
import org.zhiqim.kernel.util.Asserts;
import org.zhiqim.kernel.util.DateTimes;
import org.zhiqim.kernel.util.Images;
import org.zhiqim.kernel.util.Randoms;
import org.zhiqim.kernel.util.Resources;
import org.zhiqim.kernel.util.Sqls;
import org.zhiqim.kernel.util.Streams;
import org.zhiqim.kernel.util.Validates;
import org.zhiqim.kernel.util.codes.RSA;
import org.zhiqim.manager.dao.ZmrParamDao;
import org.zhiqim.manager.dbo.ZmrAvatar;
import org.zhiqim.manager.dbo.ZmrDept;
import org.zhiqim.manager.dbo.ZmrMenu;
import org.zhiqim.manager.dbo.ZmrOperator;
import org.zhiqim.manager.dbo.ZmrOrg;
import org.zhiqim.manager.dbo.ZmrOrgRule;
import org.zhiqim.orm.ORM;
import org.zhiqim.orm.ORMServer;
import org.zhiqim.orm.ZTable;
import org.zhiqim.orm.dbo.Selector;

/**
 * 初始化引导类
 *
 * @version v1.0.0 @author zouzhigang 2015-5-28 新建与整理
 */
public class ZmrBootstrap extends ZmrBootstrapApi
{
    private ZmrPassworder passworder;    //管理台密码器
    private ORMServer zmrServer;         //管理台独立数据库
    private ORMServer oprServer;         //操作员独立数据库
    
    /*******************************************************************************/
    //静态传入request获取属性
    /*******************************************************************************/
    
    /** 获取管理台独立数据库表指令类 */
    public static ZTable table(HttpRequest request)
    {
        return request.getContextAttribute(ZmrBootstrap.class).table();
    }
    
    /** 获取操作员独立数据库表指令类 */
    public static ZTable tableOpr(HttpRequest request)
    {
        return request.getContextAttribute(ZmrBootstrap.class).tableOpr();
    }
    
    /** 获取管理台独立数据库表指令类 */
    public static ZTable table(HttpContext context)
    {
        return context.getAttribute(ZmrBootstrap.class).table();
    }
    
    /** 获取操作员独立数据库表指令类 */
    public static ZTable tableOpr(HttpContext context)
    {
        return context.getAttribute(ZmrBootstrap.class).tableOpr();
    }
    
    /*******************************************************************************/
    //三个属性的获取
    /*******************************************************************************/
    
    /** 获取管理台密码器 */
    public ZmrPassworder getPassworder()
    {
        return passworder;
    }
    
    /** 获取管理台独立数据库表指令类 */
    public ZTable table()
    {
        return zmrServer.table();
    }
    
    /** 获取操作员独立数据库表指令类 */
    public ZTable tableOpr()
    {
        return oprServer.table();
    }
    
    /*******************************************************************************/
    //初始化
    /*******************************************************************************/
    
    @Override
    protected void initBefore() throws Exception
    {
        context.setAttribute(ZmrBootstrap.class, this);
        
        initORMServer();
        initPassworder();
    }
    
    @Override
    protected void initAfter() throws Exception
    {
        initOperator();
        initMenu();
        initParam();
        initAvatar();
        initOrg();
        initOrgRule();
        initDept();
        
        //设置主题默认值
        context.setAttribute(ZMR_THEME_INDEX, ZmrParamDao.getThemeIndex(context));
        context.setAttribute(ZMR_THEME_MAIN, ZmrParamDao.getThemeMain(context));
        
        //增加删除日志任务
        Scheduler scheduler = Global.getWithoutNew(Scheduler.class);
        scheduler.addTask(new Day(new ZmrTask(this), 5, 5, 5));
    }
    
    /** 初始化管理台独立数据库 */
    private void initORMServer()
    {
        //1.判断是否有管理台独立数据库
        String zmrOrm = context.getTopAttributeString(ZMR_ORM);
        if (Validates.isEmptyBlank(zmrOrm))
        {//未配置取默认值
            zmrServer = ORM.server(context);
        }
        else
        {//有配置必须检查正确性
            zmrServer = ORM.server(zmrOrm);
            Asserts.asserts(zmrServer != null, "上下文环境[%s]配置的管理台数据库ID[zmr.orm]未找到对应的数据库服务", context.getContextName());
        }
            
        //2.判断是否有操作员独立数据库
        String zmrOrmOpr = context.getTopAttributeString(ZMR_ORM_OPR);
        if (Validates.isEmptyBlank(zmrOrmOpr))
        {//未配置取管理台独立数据库
            oprServer = zmrServer;
        }
        else
        {//有配置必须检查正确性
            oprServer = ORM.server(zmrOrmOpr);
            Asserts.asserts(oprServer != null, "上下文环境[%s]配置的管理台操作员数据库ID[zmr.orm.opr]未找到对应的数据库服务", context.getContextName());
        }
        
    }
    
    /** 初始化密码器 */
    private void initPassworder()
    {
        String zmrPassworder = context.getTopAttributeString(ZMR_PASSWORDER, ZMR_PASSWORDER_DEFAULT);
        Object instance = Global.newInstance(zmrPassworder);
        Asserts.asserts(instance instanceof ZmrPassworder, "上下文环境[%s]的密码器配置[zmr.passworder]不是ZmrPassworder或子类", context.getContextName());
        
        passworder = (ZmrPassworder)instance;
    }
    
    /** 初始化组织，使用操作员独立数据库 */
    private void initOrg() throws Exception
    {
        if (tableOpr().count(ZmrOrg.class) > 0)
            return;
        
        ZmrOrg org = new ZmrOrg();
        org.setOrgId(ID13);
        org.setOrgName("湖南知启蒙科技有限公司");
        org.setOrgStatus(0);
        org.setOrgLevel(0);
        
        tableOpr().insert(org);
    }
    
    /** 初始操作员，使用操作员独立数据库 */
    private void initOperator() throws Exception
    {
        int count = tableOpr().count(ZmrOperator.class);
        if (count > 0)
            return;
        
        String datetime = DateTimes.getDateTimeString();
        String passSalt = Randoms.lettersDigitsSecure(64);
        
        ZmrOperator operator = new ZmrOperator();
        operator.setOrgId(ID13);
        operator.setOperatorCode(Z_NAME);
        operator.setOperatorPass(passworder.encode(Z_NAME, ZMR_PASSWORD, passSalt));
        operator.setOperatorPassSalt(passSalt);
        operator.setOperatorStatus(0);
        operator.setOperatorType(0);
        operator.setOperatorName("超级管理员");
        operator.setOperatorCreated(datetime);
        operator.setOperatorModified(datetime);
        
        table().replace(operator);
    }
    
    /** 初始化菜单 */
    private void initMenu() throws Exception
    {
        if (table().count(ZmrMenu.class) > 0)
            return;
        
        String path = context.getAttributeString(ZMR_MENU_SQL);
        if (Validates.isEmpty(path))
            return;
        
        String sqls = context.getResourceString(path);
        if (sqls == null)
            return;
        
        List<String> sqlList = Sqls.toSqlList(sqls);
        if (sqlList.isEmpty())
            return;
        
        zmrServer.sql().execute(sqlList);
    }
    
    /** 初始化参数 */
    private void initParam() throws Exception
    {
        ZmrParamDao.addParam(context, ZMR_NAME, ZMR_THEME_INDEX,             "00_default",                  1,  ItemType.PROTECTED, "首页主题（在首页主题中修改）");
        ZmrParamDao.addParam(context, ZMR_NAME, ZMR_THEME_MAIN,              "00_default",                  2,  ItemType.PROTECTED, "主页主题（在主页主题中修改）");
        ZmrParamDao.addParam(context, ZMR_NAME, ZMR_THEME_FRAME,             "false",                       3,  ItemType.PUBLIC,    "主题是否使用iframe风格");
        ZmrParamDao.addParam(context, ZMR_NAME, ZMR_HAS_MANAGE_LOGIN,        "false",                       4,  ItemType.PUBLIC,    "是否使用管理页登录退出功能");
        ZmrParamDao.addParam(context, ZMR_NAME, ZMR_HAS_VERIFICATION_CODE,   "false",                       5,  ItemType.PUBLIC,    "是否需要验证码");
        ZmrParamDao.addParam(context, ZMR_NAME, ZMR_HAS_REMEMBER_CODE,       "true",                        6,  ItemType.PUBLIC,    "是否显示记住用户名");
        ZmrParamDao.addParam(context, ZMR_NAME, ZMR_HAS_REMEMBER_PASS,       "false",                       7,  ItemType.PUBLIC,    "是否显示记住密码");
        ZmrParamDao.addParam(context, ZMR_NAME, ZMR_MAIN_URL,                "/${zhiqim_manager}/main.htm", 8,  ItemType.PUBLIC,    "登录之后的主页地址");
        ZmrParamDao.addParam(context, ZMR_NAME, ZMR_REMAIN_LOG_DAY,          "7",                           9,  ItemType.PUBLIC,    "保留操作员日志天数");
        ZmrParamDao.addParam(context, ZMR_NAME, ZMR_OPERATOR_DEPT_ALL_RULE,  "true",                        10, ItemType.PUBLIC,    "操作员权限是否继承上级部门权限");
        ZmrParamDao.addParam(context, ZMR_NAME, ZMR_CDN_PATH,                "",                            11, ItemType.PUBLIC,    "如果CDN路径不为空，JS文件优先取CDN路径");
        
        SS keys = RSA.buildKeyPair();//密钥对
        ZmrParamDao.addParam(context, ZMR_NAME, ZMR_PUBLIC_KEY,              keys.key(),                    12, ItemType.PRIVATE,   "随机生成的公钥");
        ZmrParamDao.addParam(context, ZMR_NAME, ZMR_PRIVATE_KEY,             keys.value(),                  13, ItemType.PRIVATE,   "随机生成的私钥");
    }

    /** 初始化头像 */
    private void initAvatar() throws Exception
    {
        if (table().count(ZmrAvatar.class) > 0)
            return;
        
        long avatarId = 1000000000001L;
        List<String> avatarNameList = Resources.getResourceNameList(ZmrBootstrap.class, AVATAR_PREFIX);
        for (String avatarName : avatarNameList)
        {
            byte[] data = Streams.getBytesClassPath(ZmrBootstrap.class, AVATAR_PREFIX + avatarName);
            
            byte[]  bytes50 = Images.resizePNG(data, 50, 50);
            byte[]  bytes100 = Images.resizePNG(data, 100, 100);
            byte[]  bytes150 = Images.resizePNG(data, 150, 150);
            
            ZmrAvatar avatar = new ZmrAvatar();
            avatar.setAvatarId(avatarId++);
            avatar.setAvatarType(0);
            avatar.setAvatarTime(System.currentTimeMillis());
            avatar.setAvatar50(bytes50);
            avatar.setAvatar100(bytes100);
            avatar.setAvatar150(bytes150);
            
            table().insert(avatar);
        }
    }
    
    /** 初始化组织权限 */
    private void initOrgRule() throws Exception
    {
        List<ZmrOrgRule> list = table().list(ZmrOrgRule.class,new Selector("orgId", ID13));
        if(list.size() >0)
            return;
        
        List<ZmrMenu> menuList = table().list(ZmrMenu.class);
        if(Validates.isEmpty(menuList))
            return;
        
        for(int i=0;i<menuList.size();i++)
        {
            ZmrOrgRule rule = new ZmrOrgRule();
            rule.setOrgId(ID13);
            rule.setMenuCode(menuList.get(i).getMenuCode());
            list.add(rule);
        }
        
        table().insertBatch(list);
    }
    
    /** 初始化部门 */
    private void initDept() throws Exception
    {
        if (table().count(ZmrDept.class) > 0)
            return;
        
        ZmrDept dept = new ZmrDept();
        dept.setOrgId(ID13);
        dept.setParentId(0);
        dept.setDeptId(ID13);
        dept.setDeptName("湖南知启蒙科技有限公司");
        dept.setDeptStatus(0);
        dept.setDeptLevel(0);
        dept.setDeptParentAll("0");
        
        table().insert(dept);
    }
}
